<?php
/**
 * RSS聚合系统 - 安装向导
 * 像素RPG风格，独立单文件
 */

session_start();

// 配置路径
$config_file = __DIR__ . '/config.php';
$install_file = __DIR__ . '/install.php';
$htaccess_file = __DIR__ . '/.htaccess';

// 步骤定义
$steps = ['检测', '数据库', 'SMTP', '管理员', '安装'];
$current_step = isset($_GET['step']) ? intval($_GET['step']) : 0;
if ($current_step < 0 || $current_step > 4) $current_step = 0;

// 检测是否已安装
$already_installed = file_exists($config_file);

// 处理AJAX请求
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    header('Content-Type: application/json');
    $action = $_POST['action'] ?? '';
    
    switch ($action) {
        case 'test_db':
            echo json_encode(test_database($_POST));
            exit;
        case 'test_smtp':
            echo json_encode(test_smtp($_POST));
            exit;
        case 'install':
            echo json_encode(do_install($_POST));
            exit;
        case 'force_reinstall':
            $_SESSION['force_reinstall'] = true;
            echo json_encode(['success' => true]);
            exit;
            
        case 'save_data':
            // 保存表单数据到session
            foreach ($_POST as $key => $value) {
                if (!in_array($key, ['action'])) {
                    $_SESSION['install_data'][$key] = $value;
                }
            }
            echo json_encode(['success' => true]);
            exit;
    }
}

// 环境检测函数
function check_env() {
    $checks = [];
    
    // PHP版本
    $php_version = PHP_VERSION;
    $checks['php'] = [
        'name' => 'PHP版本 >= 7.4',
        'current' => $php_version,
        'required' => '7.4.0',
        'pass' => version_compare($php_version, '7.4.0', '>=')
    ];
    
    // PDO
    $checks['pdo'] = [
        'name' => 'PDO扩展',
        'current' => extension_loaded('pdo') ? '已安装' : '未安装',
        'pass' => extension_loaded('pdo')
    ];
    
    // PDO_MYSQL
    $checks['pdo_mysql'] = [
        'name' => 'PDO_MYSQL扩展',
        'current' => extension_loaded('pdo_mysql') ? '已安装' : '未安装',
        'pass' => extension_loaded('pdo_mysql')
    ];
    
    // cURL
    $checks['curl'] = [
        'name' => 'cURL扩展',
        'current' => extension_loaded('curl') ? '已安装' : '未安装',
        'pass' => extension_loaded('curl')
    ];
    
    // XML
    $checks['xml'] = [
        'name' => 'XML/SimpleXML扩展',
        'current' => extension_loaded('simplexml') ? '已安装' : '未安装',
        'pass' => extension_loaded('simplexml')
    ];
    
    // GD（可选）
    $checks['gd'] = [
        'name' => 'GD库（验证码用）',
        'current' => extension_loaded('gd') ? '已安装' : '未安装',
        'pass' => extension_loaded('gd'),
        'optional' => true
    ];
    
    // SMTP函数（可选）
    $has_smtp = function_exists('fsockopen') || function_exists('stream_socket_client');
    $checks['smtp'] = [
        'name' => 'SMTP函数支持',
        'current' => $has_smtp ? '可用' : '受限',
        'pass' => $has_smtp,
        'optional' => true
    ];
    
    // 目录可写
    $writable = is_writable(__DIR__);
    $checks['writable'] = [
        'name' => '目录写入权限',
        'current' => $writable ? '可写' : '不可写',
        'pass' => $writable
    ];
    
    return $checks;
}

// 测试数据库连接
function test_database($data) {
    $host = $data['db_host'] ?? 'localhost';
    $name = $data['db_name'] ?? '';
    $user = $data['db_user'] ?? '';
    $pass = $data['db_pass'] ?? '';
    
    if (empty($name) || empty($user)) {
        return ['success' => false, 'error' => '数据库名和用户名不能为空'];
    }
    
    try {
        $dsn = "mysql:host=$host;charset=utf8mb4";
        $pdo = new PDO($dsn, $user, $pass);
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        
        // 检查数据库是否存在
        $stmt = $pdo->prepare("SELECT SCHEMA_NAME FROM INFORMATION_SCHEMA.SCHEMATA WHERE SCHEMA_NAME = ?");
        $stmt->execute([$name]);
        $exists = $stmt->fetch();
        
        if (!$exists) {
            // 尝试创建数据库
            $pdo->exec("CREATE DATABASE `$name` CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci");
            return ['success' => true, 'message' => '数据库不存在，已自动创建'];
        }
        
        return ['success' => true, 'message' => '连接成功，数据库已存在'];
    } catch (PDOException $e) {
        return ['success' => false, 'error' => '连接失败: ' . $e->getMessage()];
    }
}

// 测试SMTP连接
function test_smtp($data) {
    $host = $data['smtp_host'] ?? '';
    $port = intval($data['smtp_port'] ?? 587);
    $user = $data['smtp_user'] ?? '';
    $pass = $data['smtp_pass'] ?? '';
    $enc = $data['smtp_enc'] ?? 'tls';
    $test_email = $data['test_email'] ?? '';
    
    if (empty($host)) {
        return ['success' => false, 'error' => 'SMTP服务器不能为空'];
    }
    
    // 先测试连接
    $timeout = 30;
    $errno = 0;
    $errstr = '';
    
    if ($enc === 'ssl') {
        $socket = @fsockopen("ssl://$host", $port, $errno, $errstr, $timeout);
    } else {
        $socket = @fsockopen($host, $port, $errno, $errstr, $timeout);
    }
    
    if (!$socket) {
        return ['success' => false, 'error' => "无法连接到SMTP服务器: $errstr ($errno)"];
    }
    
    // 读取服务器响应
    $response = fgets($socket, 515);
    if (substr($response, 0, 3) != '220') {
        fclose($socket);
        return ['success' => false, 'error' => 'SMTP服务器响应异常: ' . $response];
    }
    
    // 发送EHLO
    fputs($socket, "EHLO " . gethostname() . "\r\n");
    while ($line = fgets($socket, 515)) {
        if (substr($line, 3, 1) == ' ') break;
    }
    
    // STARTTLS if needed
    if ($enc === 'tls') {
        fputs($socket, "STARTTLS\r\n");
        $response = fgets($socket, 515);
        if (substr($response, 0, 3) != '220') {
            fclose($socket);
            return ['success' => false, 'error' => 'STARTTLS失败: ' . $response];
        }
        stream_socket_enable_crypto($socket, true, STREAM_CRYPTO_METHOD_TLS_CLIENT);
        
        // 重新EHLO
        fputs($socket, "EHLO " . gethostname() . "\r\n");
        while ($line = fgets($socket, 515)) {
            if (substr($line, 3, 1) == ' ') break;
        }
    }
    
    // 认证
    if (!empty($user) && !empty($pass)) {
        fputs($socket, "AUTH LOGIN\r\n");
        $response = fgets($socket, 515);
        if (substr($response, 0, 3) != '334') {
            fclose($socket);
            return ['success' => false, 'error' => '认证初始化失败: ' . $response];
        }
        
        fputs($socket, base64_encode($user) . "\r\n");
        $response = fgets($socket, 515);
        if (substr($response, 0, 3) != '334') {
            fclose($socket);
            return ['success' => false, 'error' => '用户名错误: ' . $response];
        }
        
        fputs($socket, base64_encode($pass) . "\r\n");
        $response = fgets($socket, 515);
        if (substr($response, 0, 3) != '235') {
            fclose($socket);
            return ['success' => false, 'error' => '密码错误: ' . $response];
        }
    }
    
    fclose($socket);
    
    // 如果有测试邮箱，尝试发送
    if (!empty($test_email) && filter_var($test_email, FILTER_VALIDATE_EMAIL)) {
        // 这里简化处理，实际发送需要完整SMTP对话
        return ['success' => true, 'warning' => true, 'message' => '连接认证成功（测试邮件功能在系统中验证）'];
    }
    
    return ['success' => true, 'message' => 'SMTP连接和认证成功'];
}

// 执行安装
function do_install($data) {
    global $config_file, $install_file, $htaccess_file;
    
    // 收集配置
    $config = [
        'db_host' => $data['db_host'] ?? 'localhost',
        'db_name' => $data['db_name'] ?? '',
        'db_user' => $data['db_user'] ?? '',
        'db_pass' => $data['db_pass'] ?? '',
        'db_charset' => 'utf8mb4',
        'smtp_host' => $data['smtp_host'] ?? '',
        'smtp_port' => intval($data['smtp_port'] ?? 587),
        'smtp_user' => $data['smtp_user'] ?? '',
        'smtp_pass' => $data['smtp_pass'] ?? '',
        'smtp_enc' => $data['smtp_enc'] ?? 'tls',
        'admin_user' => $data['admin_user'] ?? '',
        'admin_pass' => $data['admin_pass'] ?? '',
        'admin_email' => $data['admin_email'] ?? ''
    ];
    
    if (empty($config['db_name']) || empty($config['db_user'])) {
        return ['success' => false, 'error' => '数据库配置不完整'];
    }
    
    if (empty($config['admin_user']) || empty($config['admin_pass'])) {
        return ['success' => false, 'error' => '管理员信息不完整'];
    }
    
    // 连接数据库
    try {
        $dsn = "mysql:host={$config['db_host']};charset={$config['db_charset']}";
        $pdo = new PDO($dsn, $config['db_user'], $config['db_pass']);
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        
        // 创建数据库（如果不存在）
        $pdo->exec("CREATE DATABASE IF NOT EXISTS `{$config['db_name']}` 
                    CHARACTER SET {$config['db_charset']} 
                    COLLATE {$config['db_charset']}_unicode_ci");
        $pdo->exec("USE `{$config['db_name']}`");
    } catch (PDOException $e) {
        return ['success' => false, 'error' => '数据库连接失败: ' . $e->getMessage()];
    }
    
    // 建表SQL
    $tables = get_create_tables_sql();
    
    try {
        foreach ($tables as $table => $sql) {
            $pdo->exec("DROP TABLE IF EXISTS `$table`");
            $pdo->exec($sql);
        }
    } catch (PDOException $e) {
        return ['success' => false, 'error' => '创建表失败: ' . $e->getMessage()];
    }
    
    // 插入管理员账号
    $password_hash = password_hash($config['admin_pass'], PASSWORD_DEFAULT);
    $stmt = $pdo->prepare("INSERT INTO users (username, password_hash, email, subscribe_flag, is_admin, created_at) 
                           VALUES (?, ?, ?, 1, 1, NOW())");
    $stmt->execute([$config['admin_user'], $password_hash, $config['admin_email']]);
    
    // 生成config.php
    $config_content = generate_config_file($config);
    $config_written = file_put_contents($config_file, $config_content);
    
    if (!$config_written) {
        return ['success' => false, 'error' => '无法写入配置文件，请检查目录权限'];
    }
    
    // 生成.htaccess（Apache环境）
    if (function_exists('apache_get_version') || isset($_SERVER['HTTP_HOST'])) {
        $htaccess_content = "<IfModule mod_rewrite.c>\n";
        $htaccess_content .= "RewriteEngine On\n";
        $htaccess_content .= "RewriteRule ^config\.php$ - [F,L]\n";
        $htaccess_content .= "RewriteRule ^install\.php\.bak$ - [F,L]\n";
        $htaccess_content .= "</IfModule>\n";
        $htaccess_content .= "<Files \"config.php\">\n";
        $htaccess_content .= "Order allow,deny\n";
        $htaccess_content .= "Deny from all\n";
        $htaccess_content .= "</Files>\n";
        @file_put_contents($htaccess_file, $htaccess_content);
    }
    
    // 重命名安装文件
    $renamed = @rename($install_file, $install_file . '.bak');
    
    return [
        'success' => true, 
        'message' => '安装成功！',
        'renamed' => $renamed,
        'redirect' => 'index.php'
    ];
}

// 获取建表SQL
function get_create_tables_sql() {
    return [
        // 用户表
        'users' => "CREATE TABLE users (
            id INT AUTO_INCREMENT PRIMARY KEY COMMENT '用户ID',
            username VARCHAR(32) UNIQUE NOT NULL COMMENT '用户名',
            password_hash VARCHAR(255) NOT NULL COMMENT '密码哈希',
            email VARCHAR(128) DEFAULT '' COMMENT '邮箱地址',
            subscribe_flag TINYINT DEFAULT 0 COMMENT '邮件订阅开关 0=关 1=开',
            is_admin TINYINT DEFAULT 0 COMMENT '是否管理员 0=否 1=是',
            login_fail_count INT DEFAULT 0 COMMENT '登录失败次数',
            lock_until DATETIME NULL COMMENT '锁定截止时间',
            created_at DATETIME DEFAULT CURRENT_TIMESTAMP COMMENT '创建时间',
            last_login DATETIME NULL COMMENT '最后登录时间',
            INDEX idx_email (email),
            INDEX idx_admin (is_admin)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COMMENT='用户表'",
        
        // 分类表
        'categories' => "CREATE TABLE categories (
            id INT AUTO_INCREMENT PRIMARY KEY COMMENT '分类ID',
            user_id INT NOT NULL COMMENT '所属用户',
            name VARCHAR(64) NOT NULL COMMENT '分类名称',
            color VARCHAR(7) DEFAULT '#967bb6' COMMENT '分类颜色',
            sort_order INT DEFAULT 0 COMMENT '排序',
            created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
            INDEX idx_user_sort (user_id, sort_order)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COMMENT='用户自定义分类'",
        
        // RSS源表
        'feeds' => "CREATE TABLE feeds (
            id INT AUTO_INCREMENT PRIMARY KEY COMMENT '源ID',
            user_id INT NOT NULL COMMENT '所属用户',
            plaza_id INT DEFAULT 0 COMMENT '关联广场站点ID 0=未关联',
            category_id INT DEFAULT 0 COMMENT '分类ID 0=未分类',
            title VARCHAR(128) NOT NULL COMMENT '源标题',
            feed_url VARCHAR(512) NOT NULL COMMENT 'Feed链接',
            site_url VARCHAR(512) DEFAULT '' COMMENT '站点主页',
            feed_type VARCHAR(16) DEFAULT 'rss' COMMENT '类型 rss/atom/sitemap',
            is_custom TINYINT DEFAULT 1 COMMENT '是否自定义添加 0=来自广场 1=自定义',
            description TEXT COMMENT '源描述',
            last_fetch DATETIME NULL COMMENT '最后抓取时间',
            fetch_error TEXT COMMENT '抓取错误信息',
            created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
            INDEX idx_user_category (user_id, category_id),
            INDEX idx_plaza (plaza_id),
            INDEX idx_last_fetch (last_fetch)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COMMENT='RSS订阅源'",
        
        // 广场站点表
        'plaza_sites' => "CREATE TABLE plaza_sites (
            plaza_id INT AUTO_INCREMENT PRIMARY KEY COMMENT '广场站点ID',
            title VARCHAR(128) NOT NULL COMMENT '站点标题',
            description TEXT COMMENT '站点简介',
            feed_url VARCHAR(512) NOT NULL COMMENT 'Feed链接',
            site_url VARCHAR(512) DEFAULT '' COMMENT '站点主页',
            category VARCHAR(64) DEFAULT '' COMMENT '分类标签',
            recommend_by INT DEFAULT 0 COMMENT '推荐者用户ID 0=管理员添加',
            last_update DATETIME NULL COMMENT '最后更新时间',
            article_count INT DEFAULT 0 COMMENT '文章数量',
            add_time DATETIME DEFAULT CURRENT_TIMESTAMP COMMENT '加入广场时间',
            status TINYINT DEFAULT 1 COMMENT '状态 0=禁用 1=正常',
            INDEX idx_category (category),
            INDEX idx_last_update (last_update),
            INDEX idx_status (status)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COMMENT='广场站点列表'",
        
        // 推荐审核表
        'recommendations' => "CREATE TABLE recommendations (
            recommend_id INT AUTO_INCREMENT PRIMARY KEY COMMENT '推荐ID',
            user_id INT NOT NULL COMMENT '推荐用户ID',
            feed_id INT NOT NULL COMMENT '用户Feed ID',
            title VARCHAR(128) NOT NULL COMMENT '推荐标题',
            feed_url VARCHAR(512) NOT NULL COMMENT 'Feed链接',
            reason TEXT COMMENT '推荐理由',
            status TINYINT DEFAULT 0 COMMENT '状态 0=待审核 1=已通过 2=已拒绝',
            admin_reply TEXT COMMENT '管理员回复',
            submit_time DATETIME DEFAULT CURRENT_TIMESTAMP,
            handle_time DATETIME NULL COMMENT '处理时间',
            handled_by INT DEFAULT 0 COMMENT '处理管理员ID',
            FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
            INDEX idx_status (status),
            INDEX idx_user (user_id)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COMMENT='用户推荐审核'",
        
        // 文章表
        'articles' => "CREATE TABLE articles (
            id INT AUTO_INCREMENT PRIMARY KEY COMMENT '文章ID',
            feed_id INT NOT NULL COMMENT '所属Feed',
            user_id INT NOT NULL COMMENT '所属用户',
            title VARCHAR(255) NOT NULL COMMENT '文章标题',
            link VARCHAR(1024) NOT NULL COMMENT '文章链接',
            summary TEXT COMMENT '文章摘要',
            content TEXT COMMENT '文章内容',
            author VARCHAR(128) DEFAULT '' COMMENT '作者',
            guid VARCHAR(255) COMMENT '唯一标识',
            pub_date DATETIME NULL COMMENT '发布时间',
            is_read TINYINT DEFAULT 0 COMMENT '是否已读 0=未读 1=已读',
            is_notified TINYINT DEFAULT 0 COMMENT '是否已邮件通知',
            created_at DATETIME DEFAULT CURRENT_TIMESTAMP COMMENT '抓取时间',
            FOREIGN KEY (feed_id) REFERENCES feeds(id) ON DELETE CASCADE,
            INDEX idx_feed_read (feed_id, is_read),
            INDEX idx_user_read (user_id, is_read),
            INDEX idx_pub_date (pub_date),
            INDEX idx_guid (guid),
            UNIQUE KEY uk_feed_guid (feed_id, guid(191))
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COMMENT='文章列表'",
        
        // 收藏表
        'favorites' => "CREATE TABLE favorites (
            id INT AUTO_INCREMENT PRIMARY KEY,
            user_id INT NOT NULL COMMENT '用户ID',
            article_id INT NOT NULL COMMENT '文章ID',
            created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
            FOREIGN KEY (article_id) REFERENCES articles(id) ON DELETE CASCADE,
            UNIQUE KEY uk_user_article (user_id, article_id)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COMMENT='用户收藏'",
        
        // 邮件日志表
        'email_logs' => "CREATE TABLE email_logs (
            id INT AUTO_INCREMENT PRIMARY KEY,
            user_id INT NOT NULL COMMENT '收件用户',
            article_id INT DEFAULT 0 COMMENT '关联文章ID',
            subject VARCHAR(255) COMMENT '邮件主题',
            status TINYINT DEFAULT 0 COMMENT '状态 0=失败 1=成功',
            error_msg TEXT COMMENT '错误信息',
            sent_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            INDEX idx_user (user_id),
            INDEX idx_status (status)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COMMENT='邮件发送日志'",
        
        // 登录失败日志（防暴力破解）
        'login_logs' => "CREATE TABLE login_logs (
            id INT AUTO_INCREMENT PRIMARY KEY,
            ip VARCHAR(45) NOT NULL COMMENT 'IP地址',
            username VARCHAR(32) COMMENT '尝试用户名',
            success TINYINT DEFAULT 0 COMMENT '是否成功',
            created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            INDEX idx_ip_time (ip, created_at),
            INDEX idx_created (created_at)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COMMENT='登录日志'",
        
        // 注册限制表（IP防刷）
        'register_limits' => "CREATE TABLE register_limits (
            id INT AUTO_INCREMENT PRIMARY KEY,
            ip VARCHAR(45) NOT NULL COMMENT 'IP地址',
            count INT DEFAULT 1 COMMENT '注册次数',
            last_time DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            UNIQUE KEY uk_ip (ip)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COMMENT='注册限制'"
    ];
}

// 生成配置文件内容
function generate_config_file($config) {
    // 生成随机cron密钥
    $cron_key = bin2hex(random_bytes(16));
    
    $content = "<?php\n";
    $content .= "/**\n";
    $content .= " * RSS聚合系统 - 配置文件\n";
    $content .= " * 由install.php自动生成\n";
    $content .= " */\n\n";
    $content .= "// 数据库配置\n";
    $content .= "define('DB_HOST', '{$config['db_host']}');\n";
    $content .= "define('DB_NAME', '{$config['db_name']}');\n";
    $content .= "define('DB_USER', '{$config['db_user']}');\n";
    $content .= "define('DB_PASS', '{$config['db_pass']}');\n";
    $content .= "define('DB_CHARSET', 'utf8mb4');\n\n";
    $content .= "// SMTP配置\n";
    $content .= "define('SMTP_HOST', '{$config['smtp_host']}');\n";
    $content .= "define('SMTP_PORT', {$config['smtp_port']});\n";
    $content .= "define('SMTP_USER', '{$config['smtp_user']}');\n";
    $content .= "define('SMTP_PASS', '{$config['smtp_pass']}');\n";
    $content .= "define('SMTP_ENC', '{$config['smtp_enc']}');\n";
    $content .= "define('SMTP_ENABLED', " . (empty($config['smtp_host']) ? 'false' : 'true') . ");\n\n";
    $content .= "// 安全设置\n";
    $content .= "define('LOGIN_MAX_FAIL', 3);\n";
    $content .= "define('LOGIN_LOCK_TIME', 900);\n";
    $content .= "define('REGISTER_LIMIT_PER_HOUR', 5);\n\n";
    $content .= "// 系统设置\n";
    $content .= "define('RSS_FETCH_TIMEOUT', 10);\n";
    $content .= "define('SUMMARY_LENGTH', 200);\n";
    $content .= "define('SITE_NAME', 'RSS Quest');\n";
    $content .= "define('VERSION', '1.0.0');\n";
    $content .= "define('CRON_KEY', '{$cron_key}'); // 定时任务密钥\n";
    
    return $content;
}

// 获取已保存的表单数据
function get_saved_data() {
    return $_SESSION['install_data'] ?? [];
}

function save_data($data) {
    $_SESSION['install_data'] = array_merge(get_saved_data(), $data);
}

// HTML输出
?>
<!DOCTYPE html>
<html lang="zh-CN">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>[VS] RSS Quest 安装向导 [VS]</title>
    <link href="https://fonts.googleapis.com/css2?family=Press+Start+2P&display=swap" rel="stylesheet">
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: 'Press Start 2P', 'Courier New', monospace;
            font-size: 12px;
            line-height: 1.6;
            background: #2d1b4e;
            color: #fff;
            min-height: 100vh;
            padding: 20px;
        }
        
        /* 对话框样式 */
        .dialog-box {
            max-width: 800px;
            margin: 0 auto;
            background: #1a0f2e;
            border: 4px solid #967bb6;
            padding: 20px;
            position: relative;
        }
        
        .dialog-box::before {
            content: '';
            position: absolute;
            top: -8px;
            left: -8px;
            right: -8px;
            bottom: -8px;
            border: 2px solid #4a306d;
            pointer-events: none;
        }
        
        /* 三角箭头 */
        .dialog-arrow {
            position: absolute;
            bottom: -20px;
            left: 50px;
            width: 0;
            height: 0;
            border-left: 15px solid transparent;
            border-right: 15px solid transparent;
            border-top: 20px solid #967bb6;
        }
        
        .dialog-arrow::after {
            content: '';
            position: absolute;
            top: -18px;
            left: -11px;
            border-left: 11px solid transparent;
            border-right: 11px solid transparent;
            border-top: 16px solid #1a0f2e;
        }
        
        /* 标题 */
        h1 {
            font-size: 16px;
            color: #ffd700;
            text-align: center;
            margin-bottom: 20px;
            text-shadow: 2px 2px 0 #4a306d;
        }
        
        h2 {
            font-size: 14px;
            color: #7fff00;
            margin: 15px 0 10px;
        }
        
        /* 步骤条 */
        .step-bar {
            display: flex;
            justify-content: center;
            margin-bottom: 30px;
            padding: 10px;
            background: #2d1b4e;
            border: 2px solid #967bb6;
        }
        
        .step {
            padding: 8px 12px;
            margin: 0 4px;
            background: #4a306d;
            color: #967bb6;
            font-size: 10px;
        }
        
        .step.active {
            background: #ffd700;
            color: #2d1b4e;
            animation: blink 1s infinite;
        }
        
        .step.done {
            background: #7fff00;
            color: #1a0f2e;
        }
        
        @keyframes blink {
            0%, 50% { opacity: 1; }
            51%, 100% { opacity: 0.7; }
        }
        
        /* 按钮 */
        .btn {
            display: inline-block;
            padding: 10px 20px;
            background: #967bb6;
            color: #fff;
            border: none;
            font-family: inherit;
            font-size: 12px;
            cursor: pointer;
            position: relative;
            box-shadow: 3px 3px 0 #4a306d;
            transition: all 0.1s;
            text-decoration: none;
        }
        
        .btn:hover {
            background: #b8a0d1;
        }
        
        .btn:active {
            box-shadow: -2px -2px 0 #4a306d;
            transform: translate(3px, 3px);
        }
        
        .btn.primary {
            background: #7fff00;
            color: #1a0f2e;
            box-shadow: 3px 3px 0 #4a7a00;
        }
        
        .btn.primary:hover {
            background: #9fff40;
        }
        
        .btn.danger {
            background: #ff6b6b;
            box-shadow: 3px 3px 0 #8b3a3a;
        }
        
        .btn.warning {
            background: #ffd700;
            color: #1a0f2e;
            box-shadow: 3px 3px 0 #b8a000;
        }
        
        .btn:disabled {
            opacity: 0.5;
            cursor: not-allowed;
        }
        
        /* 表单 */
        .form-group {
            margin-bottom: 15px;
        }
        
        label {
            display: block;
            margin-bottom: 8px;
            color: #ffd700;
        }
        
        input[type="text"],
        input[type="password"],
        input[type="email"],
        input[type="number"],
        select {
            width: 100%;
            padding: 10px;
            background: #1a0f2e;
            border: 3px solid #967bb6;
            color: #fff;
            font-family: inherit;
            font-size: 12px;
            outline: none;
        }
        
        input:focus {
            border-color: #ffd700;
            animation: cursor-blink 1s infinite;
        }
        
        @keyframes cursor-blink {
            0%, 49% { border-right: 2px solid #ffd700; }
            50%, 100% { border-right: 2px solid transparent; }
        }
        
        .hint {
            font-size: 10px;
            color: #967bb6;
            margin-top: 5px;
        }
        
        /* 检测结果 */
        .check-item {
            display: flex;
            align-items: center;
            padding: 10px;
            margin-bottom: 8px;
            background: #2d1b4e;
            border: 2px solid #4a306d;
        }
        
        .check-status {
            width: 24px;
            height: 24px;
            margin-right: 12px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 14px;
        }
        
        .check-status.pass {
            color: #7fff00;
        }
        
        .check-status.fail {
            color: #ff6b6b;
        }
        
        .check-status.warn {
            color: #ffd700;
        }
        
        .check-info {
            flex: 1;
        }
        
        .check-name {
            color: #fff;
        }
        
        .check-value {
            font-size: 10px;
            color: #967bb6;
            margin-top: 4px;
        }
        
        .check-value.fail {
            color: #ff6b6b;
        }
        
        /* 警告框 */
        .alert {
            padding: 15px;
            margin-bottom: 20px;
            border: 3px solid;
        }
        
        .alert-danger {
            background: #4a1a1a;
            border-color: #ff6b6b;
            color: #ff6b6b;
        }
        
        .alert-warning {
            background: #4a4000;
            border-color: #ffd700;
            color: #ffd700;
        }
        
        .alert-success {
            background: #1a4a00;
            border-color: #7fff00;
            color: #7fff00;
        }
        
        /* 进度条 */
        .progress-box {
            margin: 20px 0;
        }
        
        .progress-label {
            display: flex;
            justify-content: space-between;
            margin-bottom: 8px;
            font-size: 10px;
        }
        
        .progress-bar {
            height: 24px;
            background: #1a0f2e;
            border: 2px solid #967bb6;
            position: relative;
            overflow: hidden;
        }
        
        .progress-fill {
            height: 100%;
            background: 
                linear-gradient(90deg, #7fff00 0%, #9fff40 50%, #7fff00 100%),
                repeating-linear-gradient(
                    90deg,
                    transparent,
                    transparent 10px,
                    rgba(45, 27, 78, 0.3) 10px,
                    rgba(45, 27, 78, 0.3) 12px
                );
            width: 0%;
            transition: width 0.5s;
            box-shadow: 0 0 10px #7fff00;
        }
        
        /* 日志 */
        .install-log {
            background: #0f0a1e;
            border: 2px solid #4a306d;
            padding: 15px;
            margin-top: 20px;
            max-height: 200px;
            overflow-y: auto;
            font-size: 10px;
        }
        
        .log-item {
            padding: 4px 0;
            border-bottom: 1px dotted #4a306d;
        }
        
        .log-item.success {
            color: #7fff00;
        }
        
        .log-item.error {
            color: #ff6b6b;
        }
        
        .log-item.info {
            color: #967bb6;
        }
        
        /* 按钮组 */
        .btn-group {
            display: flex;
            gap: 10px;
            justify-content: center;
            margin-top: 20px;
        }
        
        /* 二次确认 */
        .confirm-box {
            display: none;
            padding: 15px;
            background: #4a1a1a;
            border: 3px solid #ff6b6b;
            margin-top: 15px;
            text-align: center;
        }
        
        /* 底部提示 */
        .hint-bar {
            position: fixed;
            bottom: 0;
            left: 0;
            right: 0;
            background: #1a0f2e;
            border-top: 4px solid #967bb6;
            padding: 10px 20px;
            font-size: 10px;
            color: #967bb6;
            text-align: center;
        }
        
        /* 响应式 */
        @media (max-width: 600px) {
            body {
                font-size: 10px;
                padding: 10px;
            }
            
            .step {
                font-size: 8px;
                padding: 6px 8px;
            }
            
            .btn {
                font-size: 10px;
                padding: 8px 15px;
            }
        }
        
        /* 打字效果 */
        .typing::after {
            content: '...';
            animation: dots 1.5s steps(4, end) infinite;
        }
        
        @keyframes dots {
            0%, 20% { content: ''; }
            40% { content: '.'; }
            60% { content: '..'; }
            80%, 100% { content: '...'; }
        }
        
        /* 测试结果显示 */
        .test-result {
            margin-top: 10px;
            padding: 10px;
            font-size: 10px;
            display: none;
        }
        
        .test-result.show {
            display: block;
        }
        
        .test-result.success {
            background: #1a4a00;
            border: 2px solid #7fff00;
            color: #7fff00;
        }
        
        .test-result.error {
            background: #4a1a1a;
            border: 2px solid #ff6b6b;
            color: #ff6b6b;
        }
        
        .test-result.warning {
            background: #4a4000;
            border: 2px solid #ffd700;
            color: #ffd700;
        }
        
        /* 密码强度 */
        .password-strength {
            height: 8px;
            margin-top: 8px;
            background: #1a0f2e;
            border: 1px solid #4a306d;
        }
        
        .password-strength-bar {
            height: 100%;
            width: 0%;
            transition: all 0.3s;
        }
        
        .strength-weak { background: #ff6b6b; width: 33%; }
        .strength-medium { background: #ffd700; width: 66%; }
        .strength-strong { background: #7fff00; width: 100%; }
        
        .password-hint {
            font-size: 9px;
            margin-top: 5px;
        }
    </style>
</head>
<body>
    <div class="dialog-box">
        <h1>[VS] RSS Quest 安装向导 [VS]</h1>
        
        <?php if ($already_installed && empty($_SESSION['force_reinstall'])): ?>
        <!-- 重复安装警告 -->
        <div class="alert alert-danger">
            <strong>[!] 警告！</strong><br><br>
            检测到系统已安装！<br>
            config.php 已存在<br><br>
            强制重装将清除所有数据！
        </div>
        
        <div class="btn-group">
            <a href="index.php" class="btn primary">进入系统</a>
            <button class="btn danger" onclick="showForceReinstall()">强制重装</button>
        </div>
        
        <div class="confirm-box" id="confirmBox">
            <p style="margin-bottom: 15px;">确定要清除所有数据重新安装吗？</p>
            <button class="btn" onclick="hideForceReinstall()">取消</button>
            <button class="btn danger" onclick="doForceReinstall()">确认清除</button>
        </div>
        
        <?php else: ?>
        
        <!-- 步骤条 -->
        <div class="step-bar">
            <?php foreach ($steps as $i => $name): ?>
            <span class="step <?php echo $i === $current_step ? 'active' : ($i < $current_step ? 'done' : ''); ?>">
                <?php echo $i < $current_step ? '[OK]' : ($i + 1); ?> <?php echo $name; ?>
            </span>
            <?php endforeach; ?>
        </div>
        
        <?php
        // 根据步骤显示内容
        switch ($current_step):
        case 0: // 环境检测
            $checks = check_env();
            $all_pass = true;
            $has_required_fail = false;
            foreach ($checks as $check) {
                if (!$check['pass'] && empty($check['optional'])) {
                    $all_pass = false;
                    $has_required_fail = true;
                }
            }
        ?>
        <h2>▶ 环境检测</h2>
        
        <?php foreach ($checks as $key => $check): ?>
        <div class="check-item">
            <div class="check-status <?php echo $check['pass'] ? 'pass' : (empty($check['optional']) ? 'fail' : 'warn'); ?>">
                <?php echo $check['pass'] ? '[OK]' : (empty($check['optional']) ? '[NO]' : '!'); ?>
            </div>
            <div class="check-info">
                <div class="check-name"><?php echo htmlspecialchars($check['name']); ?></div>
                <div class="check-value <?php echo !$check['pass'] && empty($check['optional']) ? 'fail' : ''; ?>">
                    <?php echo htmlspecialchars($check['current']); ?>
                    <?php if (!empty($check['optional'])): ?>
                        <span style="color: #ffd700;">（可选）</span>
                    <?php endif; ?>
                </div>
            </div>
        </div>
        <?php endforeach; ?>
        
        <div class="btn-group">
            <?php if ($has_required_fail): ?>
            <button class="btn" disabled>请先修复环境问题</button>
            <?php else: ?>
            <a href="?step=1" class="btn primary">下一步 →</a>
            <?php endif; ?>
        </div>
        
        <?php break; case 1: // 数据库配置 ?>
        <h2>▶ 数据库配置</h2>
        
        <form id="dbForm" onsubmit="return false;">
            <div class="form-group">
                <label>数据库主机</label>
                <input type="text" name="db_host" value="<?php echo get_saved_data()['db_host'] ?? 'localhost'; ?>" placeholder="localhost">
                <div class="hint">通常是 localhost 或 127.0.0.1</div>
            </div>
            
            <div class="form-group">
                <label>数据库名</label>
                <input type="text" name="db_name" value="<?php echo get_saved_data()['db_name'] ?? 'rss_quest'; ?>" placeholder="rss_quest">
                <div class="hint">如果不存在会自动创建</div>
            </div>
            
            <div class="form-group">
                <label>数据库用户名</label>
                <input type="text" name="db_user" value="<?php echo get_saved_data()['db_user'] ?? ''; ?>" placeholder="root">
            </div>
            
            <div class="form-group">
                <label>数据库密码</label>
                <input type="password" name="db_pass" value="<?php echo get_saved_data()['db_pass'] ?? ''; ?>" placeholder="">
            </div>
            
            <button type="button" class="btn warning" onclick="testDb()">[R] 测试连接</button>
            <div class="test-result" id="dbTestResult"></div>
        </form>
        
        <div class="btn-group">
            <a href="?step=0" class="btn">← 上一步</a>
            <button class="btn primary" id="dbNextBtn" onclick="goToStep(2)">下一步 →</button>
        </div>
        
        <?php break; case 2: // SMTP配置 ?>
        <h2>▶ SMTP配置（可选）</h2>
        
        <div class="alert alert-warning">
            邮件推送功能需要SMTP配置，可跳过后续在系统中设置
        </div>
        
        <form id="smtpForm" onsubmit="return false;">
            <div class="form-group">
                <label>SMTP服务器</label>
                <input type="text" name="smtp_host" value="<?php echo get_saved_data()['smtp_host'] ?? ''; ?>" placeholder="smtp.example.com">
            </div>
            
            <div class="form-group">
                <label>SMTP端口</label>
                <input type="number" name="smtp_port" value="<?php echo get_saved_data()['smtp_port'] ?? '587'; ?>" placeholder="587">
                <div class="hint">常用：25（无加密）、587（TLS）、465（SSL）</div>
            </div>
            
            <div class="form-group">
                <label>加密方式</label>
                <select name="smtp_enc">
                    <option value="tls" <?php echo (get_saved_data()['smtp_enc'] ?? 'tls') === 'tls' ? 'selected' : ''; ?>>TLS</option>
                    <option value="ssl" <?php echo (get_saved_data()['smtp_enc'] ?? '') === 'ssl' ? 'selected' : ''; ?>>SSL</option>
                    <option value="" <?php echo (get_saved_data()['smtp_enc'] ?? '') === '' ? 'selected' : ''; ?>>无加密</option>
                </select>
            </div>
            
            <div class="form-group">
                <label>SMTP用户名</label>
                <input type="text" name="smtp_user" value="<?php echo get_saved_data()['smtp_user'] ?? ''; ?>" placeholder="your@email.com">
            </div>
            
            <div class="form-group">
                <label>SMTP密码</label>
                <input type="password" name="smtp_pass" value="<?php echo get_saved_data()['smtp_pass'] ?? ''; ?>" placeholder="">
                <div class="hint">部分邮箱需使用应用专用密码</div>
            </div>
            
            <div class="form-group">
                <label>测试邮箱（可选）</label>
                <input type="email" name="test_email" placeholder="test@example.com">
            </div>
            
            <button type="button" class="btn warning" onclick="testSmtp()">[@] 测试SMTP</button>
            <div class="test-result" id="smtpTestResult"></div>
        </form>
        
        <div class="btn-group">
            <a href="?step=1" class="btn">← 上一步</a>
            <button class="btn" onclick="skipSmtp()">跳过</button>
            <button class="btn primary" onclick="goToStep(3)">下一步 →</button>
        </div>
        
        <?php break; case 3: // 管理员设置 ?>
        <h2>▶ 管理员账号</h2>
        
        <form id="adminForm" onsubmit="return false;">
            <div class="form-group">
                <label>管理员用户名</label>
                <input type="text" name="admin_user" value="<?php echo get_saved_data()['admin_user'] ?? 'admin'; ?>" placeholder="admin">
            </div>
            
            <div class="form-group">
                <label>管理员密码</label>
                <input type="password" name="admin_pass" id="adminPass" placeholder="******" oninput="checkPassword()">
                <div class="password-strength">
                    <div class="password-strength-bar" id="passStrength"></div>
                </div>
                <div class="password-hint" id="passHint"></div>
            </div>
            
            <div class="form-group">
                <label>确认密码</label>
                <input type="password" name="admin_pass2" id="adminPass2" placeholder="******" oninput="checkPasswordMatch()">
                <div class="password-hint" id="passMatchHint"></div>
            </div>
            
            <div class="form-group">
                <label>管理员邮箱</label>
                <input type="email" name="admin_email" value="<?php echo get_saved_data()['admin_email'] ?? ''; ?>" placeholder="admin@example.com">
                <div class="hint">用于接收系统通知和邮件推送</div>
            </div>
        </form>
        
        <div class="btn-group">
            <a href="?step=2" class="btn">← 上一步</a>
            <button class="btn primary" id="adminNextBtn" onclick="goToStep(4)">开始安装 →</button>
        </div>
        
        <?php break; case 4: // 安装执行 ?>
        <h2>▶ 安装中...</h2>
        
        <div class="progress-box">
            <div class="progress-label">
                <span id="progressText" class="typing">正在初始化</span>
                <span id="progressPercent">0%</span>
            </div>
            <div class="progress-bar">
                <div class="progress-fill" id="progressBar"></div>
            </div>
        </div>
        
        <div class="install-log" id="installLog">
            <div class="log-item info">[*] 安装程序已启动...</div>
        </div>
        
        <div class="btn-group" id="finishBtns" style="display: none;">
            <a href="index.php" class="btn primary">进入系统 →</a>
        </div>
        
        <div class="btn-group" id="retryBtns" style="display: none;">
            <button class="btn" onclick="location.reload()">[R] 重试</button>
        </div>
        
        <?php endswitch; endif; ?>
        
        <div class="dialog-arrow"></div>
    </div>
    
    <div class="hint-bar">
        RSS Quest v1.0 | 单文件RSS聚合系统 | PHP + MySQL
    </div>
    
    <script>
    // 保存的数据
    let installData = <?php echo json_encode(get_saved_data()); ?>;
    
    // 显示强制重装确认
    function showForceReinstall() {
        document.getElementById('confirmBox').style.display = 'block';
    }
    
    function hideForceReinstall() {
        document.getElementById('confirmBox').style.display = 'none';
    }
    
    function doForceReinstall() {
        fetch('', {
            method: 'POST',
            headers: {'Content-Type': 'application/x-www-form-urlencoded'},
            body: 'action=force_reinstall'
        })
        .then(r => r.json())
        .then(data => {
            if (data.success) location.reload();
        });
    }
    
    // 测试数据库
    function testDb() {
        const form = document.getElementById('dbForm');
        const result = document.getElementById('dbTestResult');
        const data = new FormData(form);
        data.append('action', 'test_db');
        
        result.className = 'test-result';
        result.innerHTML = '连接中...';
        result.classList.add('show');
        
        fetch('', {
            method: 'POST',
            body: new URLSearchParams(data)
        })
        .then(r => r.json())
        .then(data => {
            if (data.success) {
                result.className = 'test-result success show';
                result.innerHTML = '[OK] ' + data.message;
                saveFormData('dbForm', ['db_host', 'db_name', 'db_user', 'db_pass']);
            } else {
                result.className = 'test-result error show';
                result.innerHTML = '[NO] ' + data.error;
            }
        });
    }
    
    // 测试SMTP
    function testSmtp() {
        const form = document.getElementById('smtpForm');
        const result = document.getElementById('smtpTestResult');
        const data = new FormData(form);
        data.append('action', 'test_smtp');
        
        result.className = 'test-result';
        result.innerHTML = '连接中...';
        result.classList.add('show');
        
        fetch('', {
            method: 'POST',
            body: new URLSearchParams(data)
        })
        .then(r => r.json())
        .then(data => {
            if (data.success) {
                result.className = data.warning ? 'test-result warning show' : 'test-result success show';
                result.innerHTML = (data.warning ? '[!] ' : '[OK] ') + data.message;
                saveFormData('smtpForm', ['smtp_host', 'smtp_port', 'smtp_enc', 'smtp_user', 'smtp_pass']);
            } else {
                result.className = 'test-result error show';
                result.innerHTML = '[NO] ' + data.error;
            }
        });
    }
    
    // 跳过SMTP
    function skipSmtp() {
        // 标记SMTP为已跳过（空值）
        installData.smtp_host = '';
        installData.smtp_port = '587';
        installData.smtp_enc = 'tls';
        installData.smtp_user = '';
        installData.smtp_pass = '';
        
        // 保存到session并跳转
        fetch('', {
            method: 'POST',
            headers: {'Content-Type': 'application/x-www-form-urlencoded'},
            body: new URLSearchParams({...installData, action: 'save_data'})
        }).then(() => {
            location.href = '?step=3';
        });
    }
    
    // 保存表单数据到session
    function saveFormData(formId, fields) {
        const form = document.getElementById(formId);
        fields.forEach(field => {
            installData[field] = form.elements[field].value;
        });
    }
    
    // 检查密码强度
    function checkPassword() {
        const pass = document.getElementById('adminPass').value;
        const bar = document.getElementById('passStrength');
        const hint = document.getElementById('passHint');
        
        let strength = 0;
        if (pass.length >= 6) strength++;
        if (pass.length >= 10) strength++;
        if (/[a-z]/.test(pass) && /[A-Z]/.test(pass)) strength++;
        if (/\d/.test(pass)) strength++;
        if (/[^a-zA-Z0-9]/.test(pass)) strength++;
        
        bar.className = 'password-strength-bar';
        if (strength <= 2) {
            bar.classList.add('strength-weak');
            hint.innerHTML = '<span style="color:#ff6b6b">[!] 密码较弱，建议增加长度和复杂度</span>';
        } else if (strength <= 4) {
            bar.classList.add('strength-medium');
            hint.innerHTML = '<span style="color:#ffd700">密码强度中等</span>';
        } else {
            bar.classList.add('strength-strong');
            hint.innerHTML = '<span style="color:#7fff00">[OK] 密码强度优秀</span>';
        }
        
        checkPasswordMatch();
    }
    
    function checkPasswordMatch() {
        const pass = document.getElementById('adminPass').value;
        const pass2 = document.getElementById('adminPass2').value;
        const hint = document.getElementById('passMatchHint');
        
        if (pass2 === '') {
            hint.innerHTML = '';
        } else if (pass === pass2) {
            hint.innerHTML = '<span style="color:#7fff00">[OK] 密码一致</span>';
        } else {
            hint.innerHTML = '<span style="color:#ff6b6b">[NO] 密码不一致</span>';
        }
    }
    
    // 获取当前步骤
    function getCurrentStep() {
        const urlParams = new URLSearchParams(window.location.search);
        return parseInt(urlParams.get('step')) || 0;
    }
    
    // 前往下一步
    function goToStep(targetStep) {
        const currentStep = getCurrentStep();
        
        // 保存当前步骤的表单数据
        if (currentStep === 1) {
            // 从数据库配置页面离开，保存数据库表单
            saveFormData('dbForm', ['db_host', 'db_name', 'db_user', 'db_pass']);
        } else if (currentStep === 2) {
            // 从SMTP配置页面离开，保存SMTP表单
            saveFormData('smtpForm', ['smtp_host', 'smtp_port', 'smtp_enc', 'smtp_user', 'smtp_pass']);
        } else if (currentStep === 3) {
            // 从管理员设置页面离开，保存管理员表单
            const form = document.getElementById('adminForm');
            if (form) {
                const pass = form.elements['admin_pass'].value;
                const pass2 = form.elements['admin_pass2'].value;
                
                if (pass === '') {
                    alert('请输入管理员密码');
                    return;
                }
                if (pass !== pass2) {
                    alert('两次输入的密码不一致');
                    return;
                }
                
                saveFormData('adminForm', ['admin_user', 'admin_pass', 'admin_email']);
            }
        }
        
        // 步骤4是安装执行页面，需要发送安装请求
        if (targetStep === 4) {
            // 发送安装请求
            const allData = {...installData, action: 'install'};
            
            fetch('', {
                method: 'POST',
                headers: {'Content-Type': 'application/x-www-form-urlencoded'},
                body: new URLSearchParams(allData)
            })
            .then(r => r.json())
            .then(data => {
                if (data.success) {
                    // 安装成功，直接跳转到系统（install.php可能已被重命名）
                    window.location.href = data.redirect || 'index.php';
                } else {
                    // 安装失败，显示错误（不重命名文件，可以重试）
                    window.location.href = '?step=4&error=' + encodeURIComponent(data.error);
                }
            })
            .catch(err => {
                alert('安装请求失败: ' + err.message);
            });
            return;
        }
        
        // 保存数据到session
        fetch('', {
            method: 'POST',
            headers: {'Content-Type': 'application/x-www-form-urlencoded'},
            body: new URLSearchParams({...installData, action: 'save_data'})
        }).then(() => {
            location.href = '?step=' + targetStep;
        });
    }
    
    // 步骤4：安装执行
    <?php if ($current_step === 4): ?>
    (function() {
        const log = document.getElementById('installLog');
        const bar = document.getElementById('progressBar');
        const text = document.getElementById('progressText');
        const percent = document.getElementById('progressPercent');
        
        function addLog(msg, type = 'info') {
            const div = document.createElement('div');
            div.className = 'log-item ' + type;
            div.textContent = msg;
            log.appendChild(div);
            log.scrollTop = log.scrollHeight;
        }
        
        function setProgress(p, msg) {
            bar.style.width = p + '%';
            percent.textContent = p + '%';
            if (msg) {
                text.textContent = msg;
                text.classList.add('typing');
            }
        }
        
        // 执行安装
        function doInstall() {
            setProgress(10, '正在创建数据库表...');
            addLog('[*] 开始创建数据表...', 'info');
            
            const allData = {...installData, action: 'install'};
            
            fetch('', {
                method: 'POST',
                headers: {'Content-Type': 'application/x-www-form-urlencoded'},
                body: new URLSearchParams(allData)
            })
            .then(r => r.json())
            .then(data => {
                if (data.success) {
                    setProgress(100, '安装完成！');
                    addLog('[OK] 数据库表创建成功', 'success');
                    addLog('[OK] 管理员账号已创建', 'success');
                    addLog('[OK] 配置文件已生成', 'success');
                    if (data.renamed) {
                        addLog('[OK] 安装文件已安全处理', 'success');
                    } else {
                        addLog('[!] 请手动删除 install.php', 'warning');
                    }
                    addLog('[*] 即将跳转到系统...', 'info');
                    
                    text.classList.remove('typing');
                    text.textContent = '安装成功！';
                    document.getElementById('finishBtns').style.display = 'flex';
                    
                    setTimeout(() => {
                        window.location.href = 'index.php';
                    }, 2000);
                } else {
                    setProgress(0, '安装失败');
                    addLog('[NO] ' + data.error, 'error');
                    text.classList.remove('typing');
                    document.getElementById('retryBtns').style.display = 'flex';
                }
            })
            .catch(err => {
                setProgress(0, '安装失败');
                addLog('[NO] 请求失败: ' + err.message, 'error');
                document.getElementById('retryBtns').style.display = 'flex';
            });
        }
        
        // 页面加载后开始安装
        setTimeout(doInstall, 500);
    })();
    <?php endif; ?>
    </script>
</body>
</html>
