<?php
/**
 * RSS Quest - 单文件RSS聚合系统
 * 像素RPG风格，支持多用户、广场、邮件推送
 */

session_start();

// ==================== 基础配置与初始化 ====================

// 加载配置文件
$config_file = __DIR__ . '/config.php';
if (!file_exists($config_file)) {
    header('Location: install.php');
    exit;
}

require_once $config_file;

// 数据库连接
function get_db() {
    static $pdo = null;
    if ($pdo === null) {
        $dsn = "mysql:host=" . DB_HOST . ";dbname=" . DB_NAME . ";charset=" . DB_CHARSET;
        $pdo = new PDO($dsn, DB_USER, DB_PASS);
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        $pdo->setAttribute(PDO::ATTR_DEFAULT_FETCH_MODE, PDO::FETCH_ASSOC);
    }
    return $pdo;
}

// ==================== 工具函数 ====================

// 安全输出
function e($str) {
    return htmlspecialchars($str ?? '', ENT_QUOTES, 'UTF-8');
}

// 获取客户端IP
function get_client_ip() {
    $ip = $_SERVER['HTTP_X_FORWARDED_FOR'] ?? $_SERVER['REMOTE_ADDR'] ?? '0.0.0.0';
    return filter_var($ip, FILTER_VALIDATE_IP) ? $ip : '0.0.0.0';
}

// 检查登录状态
function is_logged_in() {
    return !empty($_SESSION['user_id']);
}

// 检查管理员
function is_admin() {
    return !empty($_SESSION['is_admin']);
}

// 需要登录
function require_login() {
    if (!is_logged_in()) {
        header('Location: ?page=login');
        exit;
    }
}

// 需要管理员
function require_admin() {
    require_login();
    if (!is_admin()) {
        header('Location: ?page=dashboard');
        exit;
    }
}

// 获取当前用户信息
function get_current_user_info() {
    if (!is_logged_in()) return null;
    $db = get_db();
    $stmt = $db->prepare("SELECT * FROM users WHERE id = ?");
    $stmt->execute([$_SESSION['user_id']]);
    return $stmt->fetch();
}

// 获取用户统计
function get_user_stats($user_id) {
    $db = get_db();
    $stats = [];
    
    // Feed数量
    $stmt = $db->prepare("SELECT COUNT(*) FROM feeds WHERE user_id = ?");
    $stmt->execute([$user_id]);
    $stats['feeds'] = $stmt->fetchColumn();
    
    // 未读文章
    $stmt = $db->prepare("SELECT COUNT(*) FROM articles WHERE user_id = ? AND is_read = 0");
    $stmt->execute([$user_id]);
    $stats['unread'] = $stmt->fetchColumn();
    
    // 收藏数
    $stmt = $db->prepare("SELECT COUNT(*) FROM favorites WHERE user_id = ?");
    $stmt->execute([$user_id]);
    $stats['favorites'] = $stmt->fetchColumn();
    
    return $stats;
}

// ==================== 安全相关 ====================

// 记录登录日志
function log_login($username, $success) {
    $db = get_db();
    $stmt = $db->prepare("INSERT INTO login_logs (ip, username, success, created_at) VALUES (?, ?, ?, NOW())");
    $stmt->execute([get_client_ip(), $username, $success ? 1 : 0]);
}

// 检查IP是否被锁定
function is_ip_locked($username) {
    $db = get_db();
    $ip = get_client_ip();
    
    // 检查该IP对该用户的失败次数
    $stmt = $db->prepare("SELECT COUNT(*) FROM login_logs 
                         WHERE ip = ? AND username = ? AND success = 0 
                         AND created_at > DATE_SUB(NOW(), INTERVAL 15 MINUTE)");
    $stmt->execute([$ip, $username]);
    $fail_count = $stmt->fetchColumn();
    
    return $fail_count >= LOGIN_MAX_FAIL;
}

// 检查注册限制
function check_register_limit() {
    $db = get_db();
    $ip = get_client_ip();
    
    // 获取或创建记录
    $stmt = $db->prepare("SELECT * FROM register_limits WHERE ip = ?");
    $stmt->execute([$ip]);
    $record = $stmt->fetch();
    
    if ($record) {
        // 检查是否超过1小时
        $last_time = strtotime($record['last_time']);
        if (time() - $last_time > 3600) {
            // 重置
            $db->prepare("UPDATE register_limits SET count = 0, last_time = NOW() WHERE ip = ?")
               ->execute([$ip]);
            return true;
        }
        return $record['count'] < REGISTER_LIMIT_PER_HOUR;
    }
    
    return true;
}

// 增加注册计数
function increment_register_count() {
    $db = get_db();
    $ip = get_client_ip();
    
    $db->prepare("INSERT INTO register_limits (ip, count, last_time) VALUES (?, 1, NOW())
                 ON DUPLICATE KEY UPDATE count = count + 1, last_time = NOW()")
       ->execute([$ip]);
}

// ==================== 验证码 ====================

// 生成一位数加减法验证码
function generate_captcha() {
    $a = rand(1, 9);
    $b = rand(1, 9);
    $is_add = rand(0, 1) === 1;
    
    if ($is_add) {
        $_SESSION['captcha_answer'] = $a + $b;
        $_SESSION['captcha_question'] = "$a + $b = ?";
    } else {
        // 确保结果为正
        if ($a < $b) { $t = $a; $a = $b; $b = $t; }
        $_SESSION['captcha_answer'] = $a - $b;
        $_SESSION['captcha_question'] = "$a - $b = ?";
    }
    
    return $_SESSION['captcha_question'];
}

// 验证验证码
function verify_captcha($answer) {
    $correct = isset($_SESSION['captcha_answer']) ? $_SESSION['captcha_answer'] : null;
    unset($_SESSION['captcha_answer']);
    unset($_SESSION['captcha_question']);
    return intval($answer) === $correct;
}

// 输出验证码图片（像素风格）
function output_captcha_image() {
    $width = 120;
    $height = 40;
    
    $image = imagecreatetruecolor($width, $height);
    
    // 背景色（深紫）
    $bg_color = imagecolorallocate($image, 45, 27, 78);
    imagefill($image, 0, 0, $bg_color);
    
    // 边框（浅紫）
    $border_color = imagecolorallocate($image, 150, 123, 182);
    imagerectangle($image, 0, 0, $width-1, $height-1, $border_color);
    imagerectangle($image, 2, 2, $width-3, $height-3, $border_color);
    
    // 文字
    $text = $_SESSION['captcha_question'] ?? '1+1=?';
    $text_color = imagecolorallocate($image, 255, 215, 0); // 金色
    
    // 像素扭曲效果
    $font_size = 5;
    $x = 15;
    $y = 12;
    
    // 绘制扭曲的文字（通过逐个像素绘制）
    imagestring($image, $font_size, $x, $y, $text, $text_color);
    
    // 添加噪点
    for ($i = 0; $i < 50; $i++) {
        $px = rand(0, $width-1);
        $py = rand(0, $height-1);
        $noise_color = imagecolorallocate($image, rand(100, 200), rand(80, 150), rand(120, 180));
        imagesetpixel($image, $px, $py, $noise_color);
    }
    
    // 添加干扰线
    for ($i = 0; $i < 3; $i++) {
        $line_color = imagecolorallocate($image, 100, 80, 120);
        imageline($image, rand(0, $width), rand(0, $height), rand(0, $width), rand(0, $height), $line_color);
    }
    
    header('Content-Type: image/png');
    imagepng($image);
    imagedestroy($image);
}

// ==================== SMTP邮件发送 ====================

function send_email($to, $subject, $body, $is_html = false) {
    if (!SMTP_ENABLED || empty(SMTP_HOST)) {
        return ['success' => false, 'error' => 'SMTP未配置'];
    }
    
    $host = SMTP_HOST;
    $port = SMTP_PORT;
    $user = SMTP_USER;
    $pass = SMTP_PASS;
    $enc = SMTP_ENC;
    
    $timeout = 30;
    $errno = 0;
    $errstr = '';
    
    // 连接
    if ($enc === 'ssl') {
        $socket = @fsockopen("ssl://$host", $port, $errno, $errstr, $timeout);
    } else {
        $socket = @fsockopen($host, $port, $errno, $errstr, $timeout);
    }
    
    if (!$socket) {
        return ['success' => false, 'error' => "连接失败: $errstr ($errno)"];
    }
    
    stream_set_timeout($socket, $timeout);
    
    // 读取欢迎信息
    $response = fgets($socket, 515);
    if (substr($response, 0, 3) != '220') {
        fclose($socket);
        return ['success' => false, 'error' => '服务器响应异常'];
    }
    
    // EHLO
    fputs($socket, "EHLO " . gethostname() . "\r\n");
    while ($line = fgets($socket, 515)) {
        if (substr($line, 3, 1) == ' ') break;
    }
    
    // STARTTLS
    if ($enc === 'tls') {
        fputs($socket, "STARTTLS\r\n");
        fgets($socket, 515);
        stream_socket_enable_crypto($socket, true, STREAM_CRYPTO_METHOD_TLS_CLIENT);
        fputs($socket, "EHLO " . gethostname() . "\r\n");
        while ($line = fgets($socket, 515)) {
            if (substr($line, 3, 1) == ' ') break;
        }
    }
    
    // 认证
    if (!empty($user) && !empty($pass)) {
        fputs($socket, "AUTH LOGIN\r\n");
        fgets($socket, 515);
        fputs($socket, base64_encode($user) . "\r\n");
        fgets($socket, 515);
        fputs($socket, base64_encode($pass) . "\r\n");
        $response = fgets($socket, 515);
        if (substr($response, 0, 3) != '235') {
            fclose($socket);
            return ['success' => false, 'error' => '认证失败'];
        }
    }
    
    // 发件人
    fputs($socket, "MAIL FROM: <$user>\r\n");
    fgets($socket, 515);
    
    // 收件人
    fputs($socket, "RCPT TO: <$to>\r\n");
    fgets($socket, 515);
    
    // 数据
    fputs($socket, "DATA\r\n");
    fgets($socket, 515);
    
    // 邮件内容
    $boundary = md5(time());
    $headers = "From: " . SITE_NAME . " <$user>\r\n";
    $headers .= "To: $to\r\n";
    $headers .= "Subject: =?UTF-8?B?" . base64_encode($subject) . "?=\r\n";
    $headers .= "MIME-Version: 1.0\r\n";
    
    if ($is_html) {
        $headers .= "Content-Type: text/html; charset=UTF-8\r\n";
    } else {
        $headers .= "Content-Type: text/plain; charset=UTF-8\r\n";
    }
    
    $message = $headers . "\r\n" . $body . "\r\n.\r\n";
    fputs($socket, $message);
    $response = fgets($socket, 515);
    
    fputs($socket, "QUIT\r\n");
    fclose($socket);
    
    if (substr($response, 0, 3) == '250' || substr($response, 0, 3) == '354') {
        return ['success' => true];
    }
    
    return ['success' => false, 'error' => '发送失败: ' . $response];
}

// 记录邮件日志
function log_email($user_id, $article_id, $subject, $success, $error = '') {
    $db = get_db();
    $stmt = $db->prepare("INSERT INTO email_logs (user_id, article_id, subject, status, error_msg, sent_at) 
                         VALUES (?, ?, ?, ?, ?, NOW())");
    $stmt->execute([$user_id, $article_id, $subject, $success ? 1 : 0, $error]);
}

// 发送新文章通知（单篇）
function notify_new_article($user, $article, $feed_title) {
    if (empty($user['email']) || !$user['subscribe_flag']) {
        return ['success' => false, 'error' => '未订阅'];
    }
    
    $subject = "【" . SITE_NAME . "】$feed_title - {$article['title']}";
    
    $body = "你好 {$user['username']},\n\n";
    $body .= "你订阅的 RSS 源有新的文章:\n\n";
    $body .= "站点: $feed_title\n";
    $body .= "标题: {$article['title']}\n";
    $body .= "时间: " . ($article['pub_date'] ?? date('Y-m-d H:i:s')) . "\n\n";
    
    if (!empty($article['summary'])) {
        $body .= "摘要:\n" . mb_substr($article['summary'], 0, 300) . "...\n\n";
    }
    
    $body .= "阅读全文: {$article['link']}\n\n";
    $body .= "--\n" . SITE_NAME;
    
    $result = send_email($user['email'], $subject, $body);
    log_email($user['id'], $article['id'] ?? 0, $subject, $result['success'], $result['error'] ?? '');
    
    return $result;
}

// 批量文章通知（多篇合并为一封邮件）
function notify_new_articles_summary($user, $articles, $feed_title, $total_count) {
    if (empty($user['email']) || !$user['subscribe_flag'] || empty($articles)) {
        return ['success' => false, 'error' => '未订阅'];
    }
    
    $article_count = count($articles);
    $more_count = $total_count - $article_count;
    
    if ($article_count === 1) {
        // 只有一篇，调用单篇通知
        return notify_new_article($user, $articles[0], $feed_title);
    }
    
    $subject = "【" . SITE_NAME . "】$feed_title - 有 {$total_count} 篇新文章";
    
    $body = "你好 {$user['username']},\n\n";
    $body .= "你订阅的 RSS 源 [$feed_title] 有 {$total_count} 篇新文章:\n\n";
    $body .= str_repeat('-', 40) . "\n\n";
    
    foreach ($articles as $i => $article) {
        $body .= ($i + 1) . ". {$article['title']}\n";
        $body .= "   时间: " . ($article['pub_date'] ?? date('Y-m-d H:i:s')) . "\n";
        if (!empty($article['summary'])) {
            $summary = mb_substr($article['summary'], 0, 150);
            if (mb_strlen($article['summary']) > 150) $summary .= '...';
            $body .= "   摘要: $summary\n";
        }
        $body .= "   链接: {$article['link']}\n\n";
    }
    
    if ($more_count > 0) {
        $body .= "... 还有 {$more_count} 篇文章，请登录系统查看全部。\n\n";
    }
    
    $body .= str_repeat('-', 40) . "\n";
    $body .= "登录查看: " . (isset($_SERVER['HTTP_HOST']) ? "https://" . $_SERVER['HTTP_HOST'] : '') . "\n";
    $body .= "--\n" . SITE_NAME;
    
    $result = send_email($user['email'], $subject, $body);
    
    // 记录日志（关联第一篇文章ID）
    log_email($user['id'], $articles[0]['id'] ?? 0, $subject, $result['success'], $result['error'] ?? '');
    
    return $result;
}

// ==================== Feed解析 ====================

function fetch_feed($url) {
    $ch = curl_init();
    curl_setopt_array($ch, [
        CURLOPT_URL => $url,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_MAXREDIRS => 3,
        CURLOPT_TIMEOUT => RSS_FETCH_TIMEOUT,
        CURLOPT_CONNECTTIMEOUT => 5, // 连接超时5秒
        CURLOPT_USERAGENT => 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36',
        CURLOPT_SSL_VERIFYPEER => false,
        CURLOPT_SSL_VERIFYHOST => false,
        CURLOPT_ACCEPT_ENCODING => '', // 自动处理gzip等压缩
        CURLOPT_MAXFILESIZE => 10 * 1024 * 1024, // 最大10MB
    ]);
    
    $content = curl_exec($ch);
    $error = curl_error($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $download_size = curl_getinfo($ch, CURLINFO_SIZE_DOWNLOAD);
    curl_close($ch);
    
    if ($content === false) {
        return ['success' => false, 'error' => $error ?: '未知错误'];
    }
    
    if ($http_code !== 200) {
        return ['success' => false, 'error' => "HTTP $http_code"];
    }
    
    // 检查内容大小
    if ($download_size > 10 * 1024 * 1024) {
        return ['success' => false, 'error' => 'Feed内容超过10MB限制'];
    }
    
    // 检查内容是否为空
    if (empty(trim($content))) {
        return ['success' => false, 'error' => 'Feed内容为空'];
    }
    
    return ['success' => true, 'content' => $content];
}

function parse_feed($content, $max_items = 100) {
    libxml_use_internal_errors(true);
    
    // 清理XML内容，防止XXE攻击和内存溢出
    $content = preg_replace('/<!DOCTYPE[^>]*>/i', '', $content);
    $content = preg_replace('/<!ENTITY[^>]*>/i', '', $content);
    
    $xml = simplexml_load_string($content);
    
    if ($xml === false) {
        return ['success' => false, 'error' => 'XML解析失败: ' . implode(', ', array_slice(libxml_get_errors(), 0, 3))];
    }
    
    $items = [];
    $feed_type = '';
    $feed_title = '';
    $feed_desc = '';
    $item_count = 0;
    
    // 检测类型
    if (isset($xml->channel)) {
        // RSS 2.0
        $feed_type = 'rss';
        $feed_title = (string)($xml->channel->title ?? '');
        $feed_desc = (string)($xml->channel->description ?? '');
        
        foreach ($xml->channel->item as $item) {
            if ($item_count >= $max_items) break;
            
            $link = (string)($item->link ?? '');
            $guid = (string)($item->guid ?? $link);
            
            // 限制摘要长度，防止内存溢出
            $summary = strip_tags((string)($item->description ?? ''));
            if (strlen($summary) > 10000) {
                $summary = mb_substr($summary, 0, 10000) . '...';
            }
            
            $items[] = [
                'title' => (string)($item->title ?? '无标题'),
                'link' => $link,
                'summary' => $summary,
                'pub_date' => parse_date((string)($item->pubDate ?? '')),
                'guid' => $guid,
                'author' => (string)($item->author ?? ''),
            ];
            $item_count++;
        }
    } elseif (isset($xml->entry)) {
        // Atom 1.0
        $feed_type = 'atom';
        $feed_title = (string)($xml->title ?? '');
        $feed_desc = (string)($xml->subtitle ?? '');
        
        foreach ($xml->entry as $entry) {
            if ($item_count >= $max_items) break;
            
            $link = '';
            if (isset($entry->link)) {
                $attrs = $entry->link->attributes();
                $link = (string)($attrs['href'] ?? '');
            }
            $id = (string)($entry->id ?? $link);
            
            $summary = '';
            if (isset($entry->content)) {
                $summary = strip_tags((string)$entry->content);
            } elseif (isset($entry->summary)) {
                $summary = strip_tags((string)$entry->summary);
            }
            
            // 限制摘要长度
            if (strlen($summary) > 10000) {
                $summary = mb_substr($summary, 0, 10000) . '...';
            }
            
            $items[] = [
                'title' => (string)($entry->title ?? '无标题'),
                'link' => $link,
                'summary' => $summary,
                'pub_date' => parse_date((string)($entry->updated ?? $entry->published ?? '')),
                'guid' => $id,
                'author' => (string)($entry->author->name ?? ''),
            ];
            $item_count++;
        }
    } elseif (isset($xml->url)) {
        // Sitemap
        $feed_type = 'sitemap';
        $feed_title = '站点地图';
        
        foreach ($xml->url as $url) {
            if ($item_count >= $max_items) break;
            
            $loc = (string)($url->loc ?? '');
            $lastmod = (string)($url->lastmod ?? '');
            
            // 尝试从URL提取标题或使用news:title
            $title = '';
            if (isset($url->{'news:news'}->{'news:title'})) {
                $title = (string)$url->{'news:news'}->{'news:title'};
            } else {
                // 从URL路径提取
                $path = parse_url($loc, PHP_URL_PATH);
                $title = basename($path) ?: '页面更新';
                $title = urldecode($title);
                $title = str_replace(['-', '_'], ' ', $title);
                $title = ucwords($title);
            }
            
            $items[] = [
                'title' => $title ?: '页面更新',
                'link' => $loc,
                'summary' => '页面更新: ' . $loc,
                'pub_date' => parse_date($lastmod),
                'guid' => $loc,
                'author' => '',
            ];
            $item_count++;
        }
    } else {
        return ['success' => false, 'error' => '无法识别的Feed格式'];
    }
    
    return [
        'success' => true,
        'type' => $feed_type,
        'title' => $feed_title,
        'description' => $feed_desc,
        'items' => $items
    ];
}

function parse_date($date_str) {
    if (empty($date_str)) {
        return date('Y-m-d H:i:s');
    }
    
    $timestamp = strtotime($date_str);
    if ($timestamp === false) {
        // 尝试Atom格式
        $date_str = preg_replace('/\.[0-9]+/', '', $date_str);
        $timestamp = strtotime($date_str);
    }
    
    return $timestamp ? date('Y-m-d H:i:s', $timestamp) : date('Y-m-d H:i:s');
}

// 抓取并保存Feed（优化版）
function fetch_and_save_feed($feed_id) {
    $db = get_db();
    $start_time = microtime(true);
    $max_execution_time = 30; // 最多执行30秒
    
    // 获取Feed信息
    $stmt = $db->prepare("SELECT * FROM feeds WHERE id = ?");
    $stmt->execute([$feed_id]);
    $feed = $stmt->fetch();
    
    if (!$feed) return ['success' => false, 'error' => 'Feed不存在'];
    
    // 获取内容
    $fetch_result = fetch_feed($feed['feed_url']);
    if (!$fetch_result['success']) {
        $db->prepare("UPDATE feeds SET fetch_error = ?, last_fetch = NOW() WHERE id = ?")
           ->execute([$fetch_result['error'], $feed_id]);
        return $fetch_result;
    }
    
    // 解析
    $parse_result = parse_feed($fetch_result['content']);
    if (!$parse_result['success']) {
        $db->prepare("UPDATE feeds SET fetch_error = ?, last_fetch = NOW() WHERE id = ?")
           ->execute([$parse_result['error'], $feed_id]);
        return $parse_result;
    }
    
    // 限制处理的文章数量（防止内存溢出）
    $max_articles = 50;
    $items = array_slice($parse_result['items'], 0, $max_articles);
    
    if (empty($items)) {
        $db->prepare("UPDATE feeds SET last_fetch = NOW(), fetch_error = NULL WHERE id = ?")
           ->execute([$feed_id]);
        return ['success' => true, 'new_count' => 0, 'type' => $parse_result['type']];
    }
    
    // 更新Feed信息
    $site_url = '';
    if (!empty($items[0]['link'])) {
        $site_url = dirname($items[0]['link']);
        if ($site_url === '.' || $site_url === '/') $site_url = $items[0]['link'];
    }
    
    $db->prepare("UPDATE feeds SET title = ?, site_url = ?, feed_type = ?, last_fetch = NOW(), fetch_error = NULL WHERE id = ?")
       ->execute([
           $parse_result['title'] ?: $feed['title'], 
           $site_url, 
           $parse_result['type'], 
           $feed_id
       ]);
    
    // 批量检查已存在的文章（减少查询次数）
    $guids = array_column($items, 'guid');
    $existing_guids = [];
    
    if (!empty($guids)) {
        // 分批查询，每批100个（避免SQL过长）
        $chunks = array_chunk($guids, 100);
        foreach ($chunks as $chunk) {
            $placeholders = implode(',', array_fill(0, count($chunk), '?'));
            $stmt = $db->prepare("SELECT guid FROM articles WHERE feed_id = ? AND guid IN ($placeholders)");
            $stmt->execute(array_merge([$feed_id], $chunk));
            while ($row = $stmt->fetch()) {
                $existing_guids[] = $row['guid'];
            }
        }
    }
    
    // 过滤出新文章
    $new_items = [];
    foreach ($items as $item) {
        if (!in_array($item['guid'], $existing_guids)) {
            $new_items[] = $item;
        }
    }
    
    $new_count = count($new_items);
    
    if ($new_count === 0) {
        return ['success' => true, 'new_count' => 0, 'type' => $parse_result['type']];
    }
    
    // 批量插入新文章（使用事务）
    $db->beginTransaction();
    try {
        $stmt = $db->prepare("INSERT INTO articles (feed_id, user_id, title, link, summary, author, guid, pub_date, created_at) 
                             VALUES (?, ?, ?, ?, ?, ?, ?, ?, NOW())");
        
        $inserted_ids = [];
        foreach ($new_items as $item) {
            // 检查超时
            if (microtime(true) - $start_time > $max_execution_time) {
                $db->rollBack();
                return ['success' => false, 'error' => '处理超时，请稍后重试'];
            }
            
            $summary = mb_substr($item['summary'], 0, SUMMARY_LENGTH);
            if (mb_strlen($item['summary']) > SUMMARY_LENGTH) {
                $summary .= '...';
            }
            
            $stmt->execute([
                $feed_id, 
                $feed['user_id'], 
                $item['title'], 
                $item['link'], 
                $summary, 
                $item['author'], 
                $item['guid'], 
                $item['pub_date']
            ]);
            
            $inserted_ids[] = [
                'id' => $db->lastInsertId(),
                'title' => $item['title'],
                'link' => $item['link'],
                'summary' => $summary,
                'pub_date' => $item['pub_date']
            ];
        }
        
        $db->commit();
        
        // 更新广场站点统计
        if ($feed['plaza_id'] > 0) {
            $db->prepare("UPDATE plaza_sites SET last_update = NOW(), article_count = article_count + ? WHERE plaza_id = ?")
               ->execute([$new_count, $feed['plaza_id']]);
        }
        
        // 发送邮件通知（限制数量，避免超时）
        // 只通知最新的3篇文章，避免邮件轰炸
        $notify_limit = 3;
        if (count($inserted_ids) > $notify_limit) {
            $notify_items = array_slice($inserted_ids, 0, $notify_limit);
        } else {
            $notify_items = $inserted_ids;
        }
        
        // 获取用户信息
        $stmt = $db->prepare("SELECT * FROM users WHERE id = ? AND subscribe_flag = 1 AND email != ''");
        $stmt->execute([$feed['user_id']]);
        $user = $stmt->fetch();
        
        if ($user && !empty($notify_items)) {
            // 如果文章很多，发送摘要邮件而不是每封单独发送
            if (count($inserted_ids) > 1) {
                notify_new_articles_summary($user, $notify_items, $parse_result['title'] ?: $feed['title'], count($inserted_ids));
            } else {
                // 只有一篇文章，单独发送
                $item = $notify_items[0];
                notify_new_article($user, $item, $parse_result['title'] ?: $feed['title']);
            }
        }
        
        // 释放大数组占用的内存
        unset($items, $new_items, $inserted_ids, $notify_items, $guids, $existing_guids);
        
    } catch (Exception $e) {
        $db->rollBack();
        // 清理内存
        unset($items, $new_items);
        return ['success' => false, 'error' => '数据库错误: ' . $e->getMessage()];
    }
    
    // 清理内存
    unset($fetch_result, $parse_result, $items);
    if (function_exists('gc_collect_cycles')) {
        gc_collect_cycles();
    }
    
    return ['success' => true, 'new_count' => $new_count, 'type' => $parse_result['type']];
}

// ==================== 页面路由处理 ====================

$page = $_GET['page'] ?? 'dashboard';
$action = $_GET['action'] ?? '';

// API/Action处理
if ($_SERVER['REQUEST_METHOD'] === 'POST' || !empty($action)) {
    header('Content-Type: application/json');
    
    switch ($action) {
        case 'captcha':
            output_captcha_image();
            exit;
            
        case 'login':
            $username = $_POST['username'] ?? '';
            $password = $_POST['password'] ?? '';
            $captcha = $_POST['captcha'] ?? '';
            
            if (!verify_captcha($captcha)) {
                echo json_encode(['success' => false, 'error' => '验证码错误']);
                exit;
            }
            
            if (is_ip_locked($username)) {
                echo json_encode(['success' => false, 'error' => '登录失败次数过多，请15分钟后再试']);
                exit;
            }
            
            $db = get_db();
            $stmt = $db->prepare("SELECT * FROM users WHERE username = ?");
            $stmt->execute([$username]);
            $user = $stmt->fetch();
            
            if ($user && password_verify($password, $user['password_hash'])) {
                // 检查账号锁定
                if ($user['lock_until'] && strtotime($user['lock_until']) > time()) {
                    $remain = ceil((strtotime($user['lock_until']) - time()) / 60);
                    log_login($username, false);
                    echo json_encode(['success' => false, 'error' => "账号已锁定，还需等待 {$remain} 分钟"]);
                    exit;
                }
                
                // 登录成功
                $_SESSION['user_id'] = $user['id'];
                $_SESSION['username'] = $user['username'];
                $_SESSION['is_admin'] = $user['is_admin'];
                
                // 重置失败次数
                $db->prepare("UPDATE users SET login_fail_count = 0, lock_until = NULL, last_login = NOW() WHERE id = ?")
                   ->execute([$user['id']]);
                
                log_login($username, true);
                echo json_encode(['success' => true]);
            } else {
                log_login($username, false);
                
                // 增加失败计数
                if ($user) {
                    $new_count = $user['login_fail_count'] + 1;
                    $lock_until = null;
                    if ($new_count >= LOGIN_MAX_FAIL) {
                        $lock_until = date('Y-m-d H:i:s', time() + LOGIN_LOCK_TIME);
                    }
                    $db->prepare("UPDATE users SET login_fail_count = ?, lock_until = ? WHERE id = ?")
                       ->execute([$new_count, $lock_until, $user['id']]);
                }
                
                echo json_encode(['success' => false, 'error' => '用户名或密码错误']);
            }
            exit;
            
        case 'register':
            $username = trim($_POST['username'] ?? '');
            $password = $_POST['password'] ?? '';
            $email = trim($_POST['email'] ?? '');
            $captcha = $_POST['captcha'] ?? '';
            
            if (!verify_captcha($captcha)) {
                echo json_encode(['success' => false, 'error' => '验证码错误']);
                exit;
            }
            
            if (!check_register_limit()) {
                echo json_encode(['success' => false, 'error' => '注册太频繁，请稍后再试']);
                exit;
            }
            
            if (!preg_match('/^[a-zA-Z0-9_]{3,20}$/', $username)) {
                echo json_encode(['success' => false, 'error' => '用户名需3-20位字母数字下划线']);
                exit;
            }
            
            if (strlen($password) < 6) {
                echo json_encode(['success' => false, 'error' => '密码至少6位']);
                exit;
            }
            
            if (!empty($email) && !filter_var($email, FILTER_VALIDATE_EMAIL)) {
                echo json_encode(['success' => false, 'error' => '邮箱格式错误']);
                exit;
            }
            
            $db = get_db();
            
            // 检查用户名
            $stmt = $db->prepare("SELECT id FROM users WHERE username = ?");
            $stmt->execute([$username]);
            if ($stmt->fetch()) {
                echo json_encode(['success' => false, 'error' => '用户名已存在']);
                exit;
            }
            
            // 创建用户
            $hash = password_hash($password, PASSWORD_DEFAULT);
            $stmt = $db->prepare("INSERT INTO users (username, password_hash, email, subscribe_flag, is_admin, created_at) 
                                 VALUES (?, ?, ?, 0, 0, NOW())");
            $stmt->execute([$username, $hash, $email]);
            
            increment_register_count();
            
            echo json_encode(['success' => true]);
            exit;
            
        case 'logout':
            session_destroy();
            echo json_encode(['success' => true]);
            exit;
            
        case 'add_feed':
            require_login();
            $url = trim($_POST['url'] ?? '');
            $category_id = intval($_POST['category_id'] ?? 0);
            
            if (!filter_var($url, FILTER_VALIDATE_URL)) {
                echo json_encode(['success' => false, 'error' => '无效的URL']);
                exit;
            }
            
            $db = get_db();
            
            // 检查是否已添加
            $stmt = $db->prepare("SELECT id FROM feeds WHERE user_id = ? AND feed_url = ?");
            $stmt->execute([$_SESSION['user_id'], $url]);
            if ($stmt->fetch()) {
                echo json_encode(['success' => false, 'error' => '该Feed已存在']);
                exit;
            }
            
            // 先测试获取
            $fetch = fetch_feed($url);
            if (!$fetch['success']) {
                echo json_encode(['success' => false, 'error' => '无法获取Feed: ' . $fetch['error']]);
                exit;
            }
            
            $parse = parse_feed($fetch['content']);
            if (!$parse['success']) {
                echo json_encode(['success' => false, 'error' => '解析失败: ' . $parse['error']]);
                exit;
            }
            
            // 插入Feed
            $db->prepare("INSERT INTO feeds (user_id, category_id, title, feed_url, site_url, feed_type, is_custom, created_at) 
                        VALUES (?, ?, ?, ?, ?, ?, 1, NOW())")
               ->execute([
                   $_SESSION['user_id'], $category_id, $parse['title'] ?: '未命名源',
                   $url, $parse['items'][0]['link'] ? dirname($parse['items'][0]['link']) : '',
                   $parse['type']
               ]);
            
            $feed_id = $db->lastInsertId();
            
            // 立即抓取文章
            fetch_and_save_feed($feed_id);
            
            echo json_encode(['success' => true, 'feed_id' => $feed_id]);
            exit;
            
        case 'delete_feed':
            require_login();
            $feed_id = intval($_POST['feed_id'] ?? 0);
            
            $db = get_db();
            $db->prepare("DELETE FROM feeds WHERE id = ? AND user_id = ?")
               ->execute([$feed_id, $_SESSION['user_id']]);
            
            echo json_encode(['success' => true]);
            exit;
            
        case 'mark_read':
            require_login();
            $article_id = intval($_POST['article_id'] ?? 0);
            $is_read = intval($_POST['is_read'] ?? 1);
            
            $db = get_db();
            $db->prepare("UPDATE articles SET is_read = ? WHERE id = ? AND user_id = ?")
               ->execute([$is_read, $article_id, $_SESSION['user_id']]);
            
            echo json_encode(['success' => true]);
            exit;
            
        case 'mark_all_read':
            require_login();
            $feed_id = intval($_GET['feed_id'] ?? 0);
            
            $db = get_db();
            $sql = "UPDATE articles SET is_read = 1 WHERE user_id = ?";
            $params = [$_SESSION['user_id']];
            if ($feed_id) {
                $sql .= " AND feed_id = ?";
                $params[] = $feed_id;
            }
            $db->prepare($sql)->execute($params);
            
            echo json_encode(['success' => true]);
            exit;
            
        case 'toggle_favorite':
            require_login();
            $article_id = intval($_POST['article_id'] ?? 0);
            
            $db = get_db();
            $stmt = $db->prepare("SELECT id FROM favorites WHERE user_id = ? AND article_id = ?");
            $stmt->execute([$_SESSION['user_id'], $article_id]);
            
            if ($stmt->fetch()) {
                $db->prepare("DELETE FROM favorites WHERE user_id = ? AND article_id = ?")
                   ->execute([$_SESSION['user_id'], $article_id]);
                echo json_encode(['success' => true, 'favorited' => false]);
            } else {
                $db->prepare("INSERT INTO favorites (user_id, article_id) VALUES (?, ?)")
                   ->execute([$_SESSION['user_id'], $article_id]);
                echo json_encode(['success' => true, 'favorited' => true]);
            }
            exit;
            
        case 'refresh_feed':
            require_login();
            $feed_id = intval($_POST['feed_id'] ?? 0);
            
            // 增加执行时间限制
            set_time_limit(120);
            ini_set('max_execution_time', 120);
            
            $db = get_db();
            $stmt = $db->prepare("SELECT * FROM feeds WHERE id = ? AND user_id = ?");
            $stmt->execute([$feed_id, $_SESSION['user_id']]);
            if (!$stmt->fetch()) {
                echo json_encode(['success' => false, 'error' => '无权操作']);
                exit;
            }
            
            // 检查Feed是否正在被更新（防止重复点击）
            $lock_key = 'feed_refresh_' . $feed_id;
            if (isset($_SESSION[$lock_key]) && (time() - $_SESSION[$lock_key]) < 30) {
                echo json_encode(['success' => false, 'error' => '正在更新中，请稍后再试']);
                exit;
            }
            $_SESSION[$lock_key] = time();
            
            $result = fetch_and_save_feed($feed_id);
            
            // 释放锁
            unset($_SESSION[$lock_key]);
            
            echo json_encode($result);
            exit;
            
        case 'add_from_plaza':
            require_login();
            $plaza_id = intval($_POST['plaza_id'] ?? 0);
            $category_id = intval($_POST['category_id'] ?? 0);
            
            $db = get_db();
            
            // 获取广场站点
            $stmt = $db->prepare("SELECT * FROM plaza_sites WHERE plaza_id = ?");
            $stmt->execute([$plaza_id]);
            $site = $stmt->fetch();
            
            if (!$site) {
                echo json_encode(['success' => false, 'error' => '站点不存在']);
                exit;
            }
            
            // 检查是否已添加
            $stmt = $db->prepare("SELECT id FROM feeds WHERE user_id = ? AND plaza_id = ?");
            $stmt->execute([$_SESSION['user_id'], $plaza_id]);
            if ($stmt->fetch()) {
                echo json_encode(['success' => false, 'error' => '已添加到自选']);
                exit;
            }
            
            // 插入
            $db->prepare("INSERT INTO feeds (user_id, plaza_id, category_id, title, feed_url, site_url, feed_type, is_custom, created_at) 
                        VALUES (?, ?, ?, ?, ?, ?, 'rss', 0, NOW())")
               ->execute([
                   $_SESSION['user_id'], $plaza_id, $category_id,
                   $site['title'], $site['feed_url'], $site['site_url']
               ]);
            
            $feed_id = $db->lastInsertId();
            fetch_and_save_feed($feed_id);
            
            echo json_encode(['success' => true]);
            exit;
            
        case 'recommend_feed':
            require_login();
            $feed_id = intval($_POST['feed_id'] ?? 0);
            $reason = trim($_POST['reason'] ?? '');
            
            $db = get_db();
            $stmt = $db->prepare("SELECT * FROM feeds WHERE id = ? AND user_id = ?");
            $stmt->execute([$feed_id, $_SESSION['user_id']]);
            $feed = $stmt->fetch();
            
            if (!$feed) {
                echo json_encode(['success' => false, 'error' => 'Feed不存在']);
                exit;
            }
            
            // 检查是否已推荐
            $stmt = $db->prepare("SELECT status FROM recommendations WHERE user_id = ? AND feed_id = ?");
            $stmt->execute([$_SESSION['user_id'], $feed_id]);
            $existing = $stmt->fetch();
            
            if ($existing) {
                if ($existing['status'] == 0) {
                    echo json_encode(['success' => false, 'error' => '已在审核中']);
                } elseif ($existing['status'] == 1) {
                    echo json_encode(['success' => false, 'error' => '已通过审核']);
                } else {
                    echo json_encode(['success' => false, 'error' => '审核未通过，请修改后重新推荐']);
                }
                exit;
            }
            
            $db->prepare("INSERT INTO recommendations (user_id, feed_id, title, feed_url, reason, status, submit_time) 
                        VALUES (?, ?, ?, ?, ?, 0, NOW())")
               ->execute([$_SESSION['user_id'], $feed_id, $feed['title'], $feed['feed_url'], $reason]);
            
            echo json_encode(['success' => true]);
            exit;
            
        case 'update_settings':
            require_login();
            $email = trim($_POST['email'] ?? '');
            $subscribe = intval($_POST['subscribe'] ?? 0);
            
            if (!empty($email) && !filter_var($email, FILTER_VALIDATE_EMAIL)) {
                echo json_encode(['success' => false, 'error' => '邮箱格式错误']);
                exit;
            }
            
            $db = get_db();
            $db->prepare("UPDATE users SET email = ?, subscribe_flag = ? WHERE id = ?")
               ->execute([$email, $subscribe, $_SESSION['user_id']]);
            
            echo json_encode(['success' => true]);
            exit;
            
        case 'change_password':
            require_login();
            $old_pass = $_POST['old_password'] ?? '';
            $new_pass = $_POST['new_password'] ?? '';
            
            if (strlen($new_pass) < 6) {
                echo json_encode(['success' => false, 'error' => '新密码至少6位']);
                exit;
            }
            
            $db = get_db();
            $stmt = $db->prepare("SELECT password_hash FROM users WHERE id = ?");
            $stmt->execute([$_SESSION['user_id']]);
            $user = $stmt->fetch();
            
            if (!password_verify($old_pass, $user['password_hash'])) {
                echo json_encode(['success' => false, 'error' => '原密码错误']);
                exit;
            }
            
            $hash = password_hash($new_pass, PASSWORD_DEFAULT);
            $db->prepare("UPDATE users SET password_hash = ? WHERE id = ?")
               ->execute([$hash, $_SESSION['user_id']]);
            
            echo json_encode(['success' => true]);
            exit;
            
        // 管理功能
        case 'admin_add_plaza':
            require_admin();
            $title = trim($_POST['title'] ?? '');
            $desc = trim($_POST['description'] ?? '');
            $url = trim($_POST['feed_url'] ?? '');
            $category = trim($_POST['category'] ?? '');
            $site_url = trim($_POST['site_url'] ?? '');
            
            if (empty($title) || empty($url)) {
                echo json_encode(['success' => false, 'error' => '标题和Feed地址必填']);
                exit;
            }
            
            $db = get_db();
            $db->prepare("INSERT INTO plaza_sites (title, description, feed_url, site_url, category, add_time) 
                        VALUES (?, ?, ?, ?, ?, NOW())")
               ->execute([$title, $desc, $url, $site_url, $category]);
            
            echo json_encode(['success' => true]);
            exit;
            
        case 'admin_delete_plaza':
            require_admin();
            $plaza_id = intval($_POST['plaza_id'] ?? 0);
            
            $db = get_db();
            $db->prepare("DELETE FROM plaza_sites WHERE plaza_id = ?")
               ->execute([$plaza_id]);
            
            echo json_encode(['success' => true]);
            exit;
            
        case 'admin_handle_recommend':
            require_admin();
            $recommend_id = intval($_POST['recommend_id'] ?? 0);
            $status = intval($_POST['status'] ?? 0); // 1=通过, 2=拒绝
            $reply = trim($_POST['reply'] ?? '');
            
            $db = get_db();
            
            if ($status == 1) {
                // 通过，添加到广场
                $stmt = $db->prepare("SELECT r.*, f.site_url FROM recommendations r 
                                    LEFT JOIN feeds f ON r.feed_id = f.id WHERE r.recommend_id = ?");
                $stmt->execute([$recommend_id]);
                $rec = $stmt->fetch();
                
                if ($rec) {
                    $db->prepare("INSERT INTO plaza_sites (title, description, feed_url, site_url, category, recommend_by, add_time) 
                                VALUES (?, ?, ?, ?, ?, ?, NOW())")
                       ->execute([$rec['title'], '', $rec['feed_url'], $rec['site_url'] ?? '', '', $rec['user_id']]);
                }
            }
            
            $db->prepare("UPDATE recommendations SET status = ?, admin_reply = ?, handle_time = NOW(), handled_by = ? WHERE recommend_id = ?")
               ->execute([$status, $reply, $_SESSION['user_id'], $recommend_id]);
            
            echo json_encode(['success' => true]);
            exit;
            
        // 后台批量刷新Feed（用于cron定时任务）
        case 'cron_refresh':
            // 需要密钥验证（防止被恶意调用）
            $cron_key = $_GET['key'] ?? '';
            if (empty(CRON_KEY) || $cron_key !== CRON_KEY) {
                echo json_encode(['success' => false, 'error' => '无效的密钥']);
                exit;
            }
            
            set_time_limit(300); // 5分钟执行时间
            ini_set('memory_limit', '256M');
            
            $db = get_db();
            
            // 获取所有需要刷新的Feed（最后更新超过1小时的）
            $stmt = $db->query("SELECT id FROM feeds 
                               WHERE last_fetch IS NULL OR last_fetch < DATE_SUB(NOW(), INTERVAL 1 HOUR)
                               ORDER BY last_fetch ASC
                               LIMIT 20");
            $feeds = $stmt->fetchAll();
            
            $results = [
                'total' => count($feeds),
                'success' => 0,
                'failed' => 0,
                'total_new_articles' => 0,
                'errors' => []
            ];
            
            foreach ($feeds as $feed) {
                $result = fetch_and_save_feed($feed['id']);
                if ($result['success']) {
                    $results['success']++;
                    $results['total_new_articles'] += $result['new_count'] ?? 0;
                } else {
                    $results['failed']++;
                    $results['errors'][] = "Feed {$feed['id']}: {$result['error']}";
                }
                
                // 每处理5个Feed暂停1秒，减轻服务器压力
                if ($results['success'] % 5 == 0) {
                    sleep(1);
                }
            }
            
            echo json_encode($results);
            exit;
            
        case 'admin_add_category':
            require_login();
            $name = trim($_POST['name'] ?? '');
            $color = trim($_POST['color'] ?? '#967bb6');
            
            if (empty($name)) {
                echo json_encode(['success' => false, 'error' => '分类名不能为空']);
                exit;
            }
            
            $db = get_db();
            $db->prepare("INSERT INTO categories (user_id, name, color, created_at) VALUES (?, ?, ?, NOW())")
               ->execute([$_SESSION['user_id'], $name, $color]);
            
            echo json_encode(['success' => true, 'id' => get_db()->lastInsertId()]);
            exit;
            
        case 'admin_delete_category':
            require_login();
            $cat_id = intval($_POST['category_id'] ?? 0);
            
            $db = get_db();
            $db->prepare("DELETE FROM categories WHERE id = ? AND user_id = ?")
               ->execute([$cat_id, $_SESSION['user_id']]);
            $db->prepare("UPDATE feeds SET category_id = 0 WHERE category_id = ? AND user_id = ?")
               ->execute([$cat_id, $_SESSION['user_id']]);
            
            echo json_encode(['success' => true]);
            exit;
    }
}

// ==================== 页面内容生成 ====================

function get_page_content($page) {
    $db = get_db();
    $user = get_current_user_info();
    $stats = $user ? get_user_stats($user['id']) : ['feeds' => 0, 'unread' => 0, 'favorites' => 0];
    
    ob_start();
    
    switch ($page) {
        case 'login':
            if (is_logged_in()) {
                header('Location: ?page=dashboard');
                exit;
            }
            generate_captcha();
            ?>
            <div class="auth-box">
                <div class="auth-tabs">
                    <span class="tab active" onclick="switchTab('login')">LOGIN</span>
                    <span class="tab" onclick="switchTab('register')">REGISTER</span>
                </div>
                
                <div id="loginForm">
                    <div class="form-group">
                        <label>用户名</label>
                        <input type="text" id="loginUser" placeholder="Enter username...">
                    </div>
                    <div class="form-group">
                        <label>密码</label>
                        <input type="password" id="loginPass" placeholder="Enter password...">
                    </div>
                    <div class="form-group">
                        <label>验证码 <span class="captcha-question"><?php echo $_SESSION['captcha_question']; ?></span></label>
                        <input type="text" id="loginCaptcha" placeholder="?" maxlength="2">
                        <img src="?action=captcha" class="captcha-img" onclick="this.src='?action=captcha&'+Math.random()" title="点击刷新">
                    </div>
                    <button class="btn primary" onclick="doLogin()">▶ START</button>
                </div>
                
                <div id="registerForm" style="display:none;">
                    <div class="form-group">
                        <label>用户名 (3-20位)</label>
                        <input type="text" id="regUser" placeholder="Username...">
                    </div>
                    <div class="form-group">
                        <label>密码 (至少6位)</label>
                        <input type="password" id="regPass" placeholder="Password...">
                    </div>
                    <div class="form-group">
                        <label>邮箱 (可选)</label>
                        <input type="email" id="regEmail" placeholder="email@example.com">
                    </div>
                    <div class="form-group">
                        <label>验证码 <span class="captcha-question"><?php echo $_SESSION['captcha_question']; ?></span></label>
                        <input type="text" id="regCaptcha" placeholder="?" maxlength="2">
                        <img src="?action=captcha" class="captcha-img" onclick="this.src='?action=captcha&'+Math.random()" title="点击刷新">
                    </div>
                    <button class="btn primary" onclick="doRegister()">▶ CREATE</button>
                </div>
            </div>
            <?php
            break;
            
        case 'dashboard':
        default:
            require_login();
            
            // 获取Feed列表
            $stmt = $db->prepare("SELECT f.*, c.name as cat_name, c.color as cat_color 
                                FROM feeds f 
                                LEFT JOIN categories c ON f.category_id = c.id 
                                WHERE f.user_id = ? ORDER BY f.created_at DESC");
            $stmt->execute([$_SESSION['user_id']]);
            $feeds = $stmt->fetchAll();
            
            // 获取分类
            $stmt = $db->prepare("SELECT * FROM categories WHERE user_id = ? ORDER BY sort_order, id");
            $stmt->execute([$_SESSION['user_id']]);
            $categories = $stmt->fetchAll();
            
            // 获取文章
            $feed_filter = intval($_GET['feed_id'] ?? 0);
            $cat_filter = intval($_GET['category_id'] ?? 0);
            $only_unread = isset($_GET['unread']) ? 1 : 0;
            
            $sql = "SELECT a.*, f.title as feed_title, f.id as feed_id,
                    EXISTS(SELECT 1 FROM favorites fav WHERE fav.article_id = a.id AND fav.user_id = ?) as is_favorited
                    FROM articles a 
                    JOIN feeds f ON a.feed_id = f.id 
                    WHERE a.user_id = ?";
            $params = [$_SESSION['user_id'], $_SESSION['user_id']];
            
            if ($feed_filter) {
                $sql .= " AND a.feed_id = ?";
                $params[] = $feed_filter;
            }
            if ($cat_filter) {
                $sql .= " AND f.category_id = ?";
                $params[] = $cat_filter;
            }
            if ($only_unread) {
                $sql .= " AND a.is_read = 0";
            }
            
            $sql .= " ORDER BY a.pub_date DESC LIMIT 50";
            
            $stmt = $db->prepare($sql);
            $stmt->execute($params);
            $articles = $stmt->fetchAll();
            ?>
            <div class="dashboard">
                <div class="sidebar">
                    <div class="panel">
                        <div class="panel-title">[F] 分类</div>
                        <div class="category-list">
                            <div class="category-item <?php echo !$cat_filter ? 'active' : ''; ?>" onclick="location.href='?page=dashboard'">
                                <span class="cat-dot" style="background:#967bb6"></span> 全部
                            </div>
                            <?php foreach ($categories as $cat): ?>
                            <div class="category-item <?php echo $cat_filter == $cat['id'] ? 'active' : ''; ?>" onclick="location.href='?page=dashboard&category_id=<?php echo $cat['id']; ?>'">
                                <span class="cat-dot" style="background:<?php echo $cat['color']; ?>"></span> <?php echo e($cat['name']); ?>
                                <span class="cat-actions" onclick="event.stopPropagation();deleteCategory(<?php echo $cat['id']; ?>)">×</span>
                            </div>
                            <?php endforeach; ?>
                        </div>
                        <button class="btn small" onclick="showAddCategory()">+ 新分类</button>
                    </div>
                    
                    <div class="panel">
                        <div class="panel-title">[R] RSS源</div>
                        <div class="feed-list">
                            <?php foreach ($feeds as $feed): ?>
                            <div class="feed-item <?php echo $feed_filter == $feed['id'] ? 'active' : ''; ?>" onclick="location.href='?page=dashboard&feed_id=<?php echo $feed['id']; ?>'">
                                <span class="feed-icon">[R]</span>
                                <span class="feed-name"><?php echo e($feed['title']); ?></span>
                                <?php if ($feed['fetch_error']): ?>
                                <span class="feed-error" title="<?php echo e($feed['fetch_error']); ?>">!</span>
                                <?php endif; ?>
                                <span class="feed-actions">
                                    <span onclick="event.stopPropagation();refreshFeed(<?php echo $feed['id']; ?>)">[R]</span>
                                    <span onclick="event.stopPropagation();deleteFeed(<?php echo $feed['id']; ?>)">×</span>
                                </span>
                            </div>
                            <?php endforeach; ?>
                        </div>
                        <button class="btn small" onclick="showAddFeed()">+ 添加源</button>
                    </div>
                </div>
                
                <div class="main-content">
                    <div class="content-header">
                        <h2><?php echo $feed_filter ? 'Feed文章' : ($cat_filter ? '分类文章' : '全部文章'); ?></h2>
                        <div class="header-actions">
                            <label class="checkbox-label">
                                <input type="checkbox" <?php echo $only_unread ? 'checked' : ''; ?> onchange="toggleUnread(this)">
                                仅未读
                            </label>
                            <?php if ($feed_filter || $cat_filter || $only_unread): ?>
                            <a href="?page=dashboard" class="btn small">显示全部</a>
                            <?php endif; ?>
                            <button class="btn small" onclick="markAllRead(<?php echo $feed_filter; ?>)">全部已读</button>
                        </div>
                    </div>
                    
                    <div class="article-list">
                        <?php if (empty($articles)): ?>
                        <div class="empty-state">
                            <div class="empty-icon">[M]</div>
                            <p>暂无文章</p>
                            <p class="hint">点击左侧"添加源"订阅RSS</p>
                        </div>
                        <?php else: ?>
                        <?php foreach ($articles as $article): ?>
                        <div class="article-item <?php echo $article['is_read'] ? 'read' : 'unread'; ?>" data-id="<?php echo $article['id']; ?>">
                            <?php if (!$article['is_read']): ?>
                            <span class="unread-badge">!</span>
                            <?php endif; ?>
                            <div class="article-header">
                                <span class="article-feed"><?php echo e($article['feed_title']); ?></span>
                                <span class="article-time"><?php echo date('m-d H:i', strtotime($article['pub_date'])); ?></span>
                            </div>
                            <a href="<?php echo e($article['link']); ?>" target="_blank" class="article-title" onclick="markRead(<?php echo $article['id']; ?>)">
                                <?php echo e($article['title']); ?>
                            </a>
                            <div class="article-summary"><?php echo e($article['summary']); ?></div>
                            <div class="article-actions">
                                <span class="action-btn <?php echo $article['is_favorited'] ? 'favorited' : ''; ?>" onclick="toggleFavorite(<?php echo $article['id']; ?>, this)">
                                    <?php echo $article['is_favorited'] ? '<3' : '</3'; ?> 收藏
                                </span>
                                <span class="action-btn" onclick="markRead(<?php echo $article['id']; ?>, true)">[OK] 已读</span>
                                <a href="<?php echo e($article['link']); ?>" target="_blank" class="action-btn">↗ 阅读</a>
                            </div>
                        </div>
                        <?php endforeach; ?>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
            <?php
            break;
            
        case 'plaza':
            // 获取广场站点（按最后更新排序）
            $stmt = $db->query("SELECT * FROM plaza_sites WHERE status = 1 ORDER BY last_update DESC, add_time DESC");
            $sites = $stmt->fetchAll();
            
            // 获取当前用户已添加的广场源
            $my_plaza_ids = [];
            if (is_logged_in()) {
                $stmt = $db->prepare("SELECT plaza_id FROM feeds WHERE user_id = ? AND plaza_id > 0");
                $stmt->execute([$_SESSION['user_id']]);
                $my_plaza_ids = $stmt->fetchAll(PDO::FETCH_COLUMN);
            }
            ?>
            <div class="plaza">
                <h2>[W] 站点广场</h2>
                <p class="plaza-desc">发现优质RSS源，一键添加到自选</p>
                
                <div class="site-grid">
                    <?php foreach ($sites as $site): 
                        // 获取最新3篇文章
                        $articles = [];
                        $stmt = $db->prepare("SELECT a.title, a.link, a.pub_date 
                                            FROM articles a 
                                            JOIN feeds f ON a.feed_id = f.id 
                                            WHERE f.plaza_id = ? 
                                            ORDER BY a.pub_date DESC LIMIT 3");
                        $stmt->execute([$site['plaza_id']]);
                        $articles = $stmt->fetchAll();
                        
                        $is_new = $site['last_update'] && (time() - strtotime($site['last_update'])) < 86400;
                        $is_added = in_array($site['plaza_id'], $my_plaza_ids);
                    ?>
                    <div class="site-card">
                        <?php if ($is_new): ?>
                        <span class="new-badge">NEW</span>
                        <?php endif; ?>
                        <div class="site-header">
                            <div class="site-icon">[S]</div>
                            <div class="site-info">
                                <h3><?php echo e($site['title']); ?></h3>
                                <span class="site-category"><?php echo e($site['category'] ?: '未分类'); ?></span>
                            </div>
                        </div>
                        <p class="site-desc"><?php echo e($site['description'] ?: '暂无描述'); ?></p>
                        <div class="site-latest">
                            <div class="latest-title">最新文章:</div>
                            <?php if (empty($articles)): ?>
                            <div class="latest-empty">暂无文章</div>
                            <?php else: ?>
                            <?php foreach ($articles as $art): ?>
                            <a href="<?php echo e($art['link']); ?>" target="_blank" class="latest-item">
                                <?php echo e(mb_substr($art['title'], 0, 30)); ?><?php echo mb_strlen($art['title']) > 30 ? '...' : ''; ?>
                            </a>
                            <?php endforeach; ?>
                            <?php endif; ?>
                        </div>
                        <div class="site-footer">
                            <?php if (is_logged_in()): ?>
                            <button class="btn <?php echo $is_added ? 'disabled' : 'primary'; ?>" 
                                    <?php echo $is_added ? 'disabled' : "onclick='addFromPlaza({$site['plaza_id']})'"; ?>>
                                <?php echo $is_added ? '[OK] 已添加' : '+ 添加'; ?>
                            </button>
                            <?php else: ?>
                            <a href="?page=login" class="btn">登录后添加</a>
                            <?php endif; ?>
                            <span class="update-time">
                                <?php echo $site['last_update'] ? date('m-d H:i', strtotime($site['last_update'])) : '未更新'; ?>
                            </span>
                        </div>
                    </div>
                    <?php endforeach; ?>
                </div>
            </div>
            <?php
            break;
            
        case 'favorites':
            require_login();
            
            $stmt = $db->prepare("SELECT a.*, f.title as feed_title,
                                EXISTS(SELECT 1 FROM favorites fav WHERE fav.article_id = a.id AND fav.user_id = ?) as is_favorited
                                FROM articles a 
                                JOIN feeds f ON a.feed_id = f.id 
                                JOIN favorites fav ON a.id = fav.article_id 
                                WHERE fav.user_id = ? 
                                ORDER BY fav.created_at DESC");
            $stmt->execute([$_SESSION['user_id'], $_SESSION['user_id']]);
            $articles = $stmt->fetchAll();
            ?>
            <div class="favorites-page">
                <h2><3 我的收藏</h2>
                
                <?php if (empty($articles)): ?>
                <div class="empty-state">
                    <div class="empty-icon"></3></div>
                    <p>还没有收藏任何文章</p>
                    <a href="?page=dashboard" class="btn">去阅读</a>
                </div>
                <?php else: ?>
                <div class="article-list">
                    <?php foreach ($articles as $article): ?>
                    <div class="article-item <?php echo $article['is_read'] ? 'read' : 'unread'; ?>">
                        <div class="article-header">
                            <span class="article-feed"><?php echo e($article['feed_title']); ?></span>
                            <span class="article-time"><?php echo date('Y-m-d H:i', strtotime($article['pub_date'])); ?></span>
                        </div>
                        <a href="<?php echo e($article['link']); ?>" target="_blank" class="article-title">
                            <?php echo e($article['title']); ?>
                        </a>
                        <div class="article-summary"><?php echo e($article['summary']); ?></div>
                        <div class="article-actions">
                            <span class="action-btn favorited" onclick="toggleFavorite(<?php echo $article['id']; ?>, this)"><3 已收藏</span>
                            <a href="<?php echo e($article['link']); ?>" target="_blank" class="action-btn">↗ 阅读</a>
                        </div>
                    </div>
                    <?php endforeach; ?>
                </div>
                <?php endif; ?>
            </div>
            <?php
            break;
            
        case 'settings':
            require_login();
            $user = get_current_user_info();
            ?>
            <div class="settings-page">
                <h2>[S] 设置</h2>
                
                <div class="settings-section">
                    <h3>邮件设置</h3>
                    <div class="form-group">
                        <label>邮箱地址</label>
                        <input type="email" id="settingEmail" value="<?php echo e($user['email']); ?>" placeholder="your@email.com">
                    </div>
                    <div class="form-group">
                        <label>
                            <input type="checkbox" id="settingSubscribe" <?php echo $user['subscribe_flag'] ? 'checked' : ''; ?>>
                            <span class="mail-icon <?php echo $user['subscribe_flag'] ? 'subscribed' : ''; ?>">
                                <?php echo $user['subscribe_flag'] ? '[@]' : '[@]+[X]'; ?>
                            </span>
                            开启邮件订阅（新文章推送）
                        </label>
                    </div>
                    <button class="btn primary" onclick="saveSettings()">保存设置</button>
                </div>
                
                <div class="settings-section">
                    <h3>修改密码</h3>
                    <div class="form-group">
                        <label>原密码</label>
                        <input type="password" id="oldPassword" placeholder="******">
                    </div>
                    <div class="form-group">
                        <label>新密码</label>
                        <input type="password" id="newPassword" placeholder="******">
                    </div>
                    <button class="btn" onclick="changePassword()">修改密码</button>
                </div>
            </div>
            <?php
            break;
            
        case 'admin':
            require_admin();
            
            $tab = $_GET['tab'] ?? 'plaza';
            ?>
            <div class="admin-page">
                <h2>[A] 管理后台</h2>
                
                <div class="admin-tabs">
                    <a href="?page=admin&tab=plaza" class="tab <?php echo $tab=='plaza'?'active':''; ?>">广场管理</a>
                    <a href="?page=admin&tab=review" class="tab <?php echo $tab=='review'?'active':''; ?>">审核队列</a>
                    <a href="?page=admin&tab=users" class="tab <?php echo $tab=='users'?'active':''; ?>">用户管理</a>
                </div>
                
                <?php if ($tab == 'plaza'): ?>
                <div class="admin-section">
                    <h3>添加站点到广场</h3>
                    <div class="form-row">
                        <input type="text" id="plazaTitle" placeholder="站点标题">
                        <input type="text" id="plazaUrl" placeholder="Feed地址">
                        <input type="text" id="plazaSiteUrl" placeholder="站点主页(可选)">
                        <input type="text" id="plazaCategory" placeholder="分类(可选)">
                    </div>
                    <textarea id="plazaDesc" placeholder="站点简介"></textarea>
                    <button class="btn primary" onclick="addPlazaSite()">添加站点</button>
                </div>
                
                <div class="admin-section">
                    <h3>广场站点列表</h3>
                    <table class="data-table">
                        <tr><th>ID</th><th>标题</th><th>分类</th><th>更新时间</th><th>操作</th></tr>
                        <?php 
                        $stmt = $db->query("SELECT * FROM plaza_sites ORDER BY plaza_id DESC");
                        while ($site = $stmt->fetch()):
                        ?>
                        <tr>
                            <td><?php echo $site['plaza_id']; ?></td>
                            <td><?php echo e($site['title']); ?></td>
                            <td><?php echo e($site['category']); ?></td>
                            <td><?php echo $site['last_update'] ? date('Y-m-d H:i', strtotime($site['last_update'])) : '未更新'; ?></td>
                            <td><button class="btn small danger" onclick="deletePlaza(<?php echo $site['plaza_id']; ?>)">删除</button></td>
                        </tr>
                        <?php endwhile; ?>
                    </table>
                </div>
                
                <?php elseif ($tab == 'review'): ?>
                <div class="admin-section">
                    <h3>用户推荐审核</h3>
                    <?php 
                    $stmt = $db->query("SELECT r.*, u.username FROM recommendations r 
                                       JOIN users u ON r.user_id = u.id 
                                       WHERE r.status = 0 ORDER BY r.submit_time DESC");
                    $recommendations = $stmt->fetchAll();
                    
                    if (empty($recommendations)): ?>
                    <p class="empty">暂无待审核推荐</p>
                    <?php else: ?>
                    <div class="review-list">
                        <?php foreach ($recommendations as $rec): ?>
                        <div class="review-item">
                            <div class="review-info">
                                <strong><?php echo e($rec['title']); ?></strong>
                                <span>推荐人: <?php echo e($rec['username']); ?></span>
                                <span>URL: <?php echo e($rec['feed_url']); ?></span>
                                <?php if ($rec['reason']): ?>
                                <span>理由: <?php echo e($rec['reason']); ?></span>
                                <?php endif; ?>
                            </div>
                            <div class="review-actions">
                                <input type="text" id="reply_<?php echo $rec['recommend_id']; ?>" placeholder="回复(可选)">
                                <button class="btn primary" onclick="handleRecommend(<?php echo $rec['recommend_id']; ?>, 1)">[OK] 通过</button>
                                <button class="btn danger" onclick="handleRecommend(<?php echo $rec['recommend_id']; ?>, 2)">[NO] 拒绝</button>
                            </div>
                        </div>
                        <?php endforeach; ?>
                    </div>
                    <?php endif; ?>
                </div>
                
                <?php elseif ($tab == 'users'): ?>
                <div class="admin-section">
                    <h3>用户列表</h3>
                    <table class="data-table">
                        <tr><th>ID</th><th>用户名</th><th>邮箱</th><th>订阅</th><th>注册时间</th></tr>
                        <?php 
                        $stmt = $db->query("SELECT * FROM users ORDER BY id DESC");
                        while ($u = $stmt->fetch()):
                        ?>
                        <tr>
                            <td><?php echo $u['id']; ?></td>
                            <td><?php echo e($u['username']); ?> <?php echo $u['is_admin'] ? '(管理员)' : ''; ?></td>
                            <td><?php echo e($u['email']); ?></td>
                            <td><?php echo $u['subscribe_flag'] ? '[OK]' : '[NO]'; ?></td>
                            <td><?php echo date('Y-m-d', strtotime($u['created_at'])); ?></td>
                        </tr>
                        <?php endwhile; ?>
                    </table>
                </div>
                <?php endif; ?>
            </div>
            <?php
            break;
    }
    
    return ob_get_clean();
}

// 获取模态框HTML
function get_modals() {
    ?>
    <!-- 添加Feed模态框 -->
    <div id="modalAddFeed" class="modal">
        <div class="modal-content">
            <h3>[R] 添加RSS源</h3>
            <div class="form-group">
                <label>Feed地址</label>
                <input type="text" id="newFeedUrl" placeholder="https://example.com/feed.xml">
            </div>
            <div class="form-group">
                <label>分类</label>
                <select id="newFeedCategory">
                    <option value="0">未分类</option>
                    <?php 
                    if (is_logged_in()) {
                        $db = get_db();
                        $stmt = $db->prepare("SELECT * FROM categories WHERE user_id = ?");
                        $stmt->execute([$_SESSION['user_id']]);
                        foreach ($stmt->fetchAll() as $cat) {
                            echo '<option value="' . $cat['id'] . '">' . e($cat['name']) . '</option>';
                        }
                    }
                    ?>
                </select>
            </div>
            <div class="modal-actions">
                <button class="btn" onclick="closeModal('modalAddFeed')">取消</button>
                <button class="btn primary" onclick="submitAddFeed()">添加</button>
            </div>
        </div>
    </div>
    
    <!-- 添加分类模态框 -->
    <div id="modalAddCategory" class="modal">
        <div class="modal-content">
            <h3>[F] 新分类</h3>
            <div class="form-group">
                <label>分类名称</label>
                <input type="text" id="newCategoryName" placeholder="分类名称">
            </div>
            <div class="form-group">
                <label>颜色</label>
                <input type="color" id="newCategoryColor" value="#967bb6">
            </div>
            <div class="modal-actions">
                <button class="btn" onclick="closeModal('modalAddCategory')">取消</button>
                <button class="btn primary" onclick="submitAddCategory()">创建</button>
            </div>
        </div>
    </div>
    <?php
}

// ==================== 输出HTML ====================
?>
<!DOCTYPE html>
<html lang="zh-CN">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo SITE_NAME; ?> - RSS Quest</title>
    <link href="https://fonts.googleapis.com/css2?family=Press+Start+2P&display=swap" rel="stylesheet">
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        
        body {
            font-family: 'Press Start 2P', 'Courier New', monospace;
            font-size: 11px;
            line-height: 1.8;
            background: #2d1b4e;
            color: #fff;
            min-height: 100vh;
        }
        
        /* 顶部状态栏 */
        .status-bar {
            background: #1a0f2e;
            border-bottom: 4px solid #967bb6;
            padding: 10px 20px;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .status-left {
            display: flex;
            gap: 20px;
        }
        
        .status-item {
            color: #7fff00;
        }
        
        .status-item span {
            color: #fff;
        }
        
        .status-mail {
            color: <?php echo (is_logged_in() && ($user = get_current_user_info()) && $user['subscribe_flag']) ? '#7fff00' : '#ff6b6b'; ?>;
        }
        
        .status-mail.subscribed {
            animation: blink 1s infinite;
        }
        
        @keyframes blink {
            0%, 50% { opacity: 1; }
            51%, 100% { opacity: 0.5; }
        }
        
        /* 导航 */
        .nav-bar {
            background: #4a306d;
            padding: 0 20px;
            display: flex;
            gap: 5px;
        }
        
        .nav-item {
            padding: 12px 20px;
            color: #967bb6;
            text-decoration: none;
            border-bottom: 4px solid transparent;
            transition: all 0.2s;
        }
        
        .nav-item:hover, .nav-item.active {
            color: #fff;
            background: #2d1b4e;
            border-bottom-color: #ffd700;
        }
        
        /* 主内容 */
        .container {
            max-width: 1400px;
            margin: 0 auto;
            padding: 20px;
        }
        
        /* 对话框样式 */
        .dialog-box {
            background: #1a0f2e;
            border: 4px solid #967bb6;
            padding: 20px;
            position: relative;
            margin-bottom: 20px;
        }
        
        .dialog-box::before {
            content: '';
            position: absolute;
            top: -8px; left: -8px; right: -8px; bottom: -8px;
            border: 2px solid #4a306d;
            pointer-events: none;
        }
        
        /* 按钮 */
        .btn {
            display: inline-block;
            padding: 10px 20px;
            background: #967bb6;
            color: #fff;
            border: none;
            font-family: inherit;
            font-size: 10px;
            cursor: pointer;
            box-shadow: 3px 3px 0 #4a306d;
            text-decoration: none;
        }
        
        .btn:hover { background: #b8a0d1; }
        .btn:active { box-shadow: -2px -2px 0 #4a306d; transform: translate(3px, 3px); }
        .btn.primary { background: #7fff00; color: #1a0f2e; box-shadow: 3px 3px 0 #4a7a00; }
        .btn.primary:hover { background: #9fff40; }
        .btn.danger { background: #ff6b6b; box-shadow: 3px 3px 0 #8b3a3a; }
        .btn.small { padding: 6px 12px; font-size: 9px; }
        .btn.disabled { opacity: 0.5; cursor: not-allowed; }
        
        /* 表单 */
        .form-group { margin-bottom: 15px; }
        label { display: block; margin-bottom: 8px; color: #ffd700; font-size: 10px; }
        input, select, textarea {
            width: 100%;
            padding: 10px;
            background: #0f0a1e;
            border: 3px solid #967bb6;
            color: #fff;
            font-family: inherit;
            font-size: 10px;
        }
        input:focus, select:focus, textarea:focus { border-color: #ffd700; outline: none; }
        textarea { min-height: 80px; resize: vertical; }
        
        /* 登录/注册 */
        .auth-box {
            max-width: 400px;
            margin: 50px auto;
            background: #1a0f2e;
            border: 4px solid #967bb6;
            padding: 30px;
        }
        .auth-tabs { display: flex; margin-bottom: 20px; border-bottom: 2px solid #4a306d; }
        .tab { padding: 10px 20px; color: #967bb6; cursor: pointer; }
        .tab.active { color: #ffd700; border-bottom: 3px solid #ffd700; }
        .captcha-question { color: #7fff00; margin-left: 10px; }
        .captcha-img { display: block; margin-top: 10px; cursor: pointer; border: 2px solid #967bb6; }
        
        /* Dashboard */
        .dashboard { display: flex; gap: 20px; }
        .sidebar { width: 250px; flex-shrink: 0; }
        .main-content { flex: 1; }
        
        .panel {
            background: #1a0f2e;
            border: 3px solid #967bb6;
            padding: 15px;
            margin-bottom: 20px;
        }
        .panel-title {
            color: #ffd700;
            margin-bottom: 15px;
            padding-bottom: 10px;
            border-bottom: 2px solid #4a306d;
        }
        
        .category-list, .feed-list { margin-bottom: 10px; }
        .category-item, .feed-item {
            padding: 8px;
            margin-bottom: 5px;
            cursor: pointer;
            display: flex;
            align-items: center;
            gap: 8px;
        }
        .category-item:hover, .feed-item:hover { background: #2d1b4e; }
        .category-item.active { background: #4a306d; }
        .cat-dot { width: 10px; height: 10px; display: inline-block; }
        .cat-actions { margin-left: auto; color: #ff6b6b; cursor: pointer; display: none; }
        .category-item:hover .cat-actions { display: block; }
        
        .feed-icon { font-size: 12px; }
        .feed-name { flex: 1; overflow: hidden; text-overflow: ellipsis; white-space: nowrap; }
        .feed-error { color: #ff6b6b; font-weight: bold; }
        .feed-actions { display: none; gap: 5px; }
        .feed-item:hover .feed-actions { display: flex; }
        
        .content-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 20px;
            padding-bottom: 15px;
            border-bottom: 2px solid #967bb6;
        }
        .content-header h2 { color: #ffd700; }
        .header-actions { display: flex; gap: 10px; align-items: center; }
        .checkbox-label { display: flex; align-items: center; gap: 5px; cursor: pointer; }
        
        /* 文章列表 */
        .article-list { display: flex; flex-direction: column; gap: 15px; }
        .article-item {
            background: #1a0f2e;
            border: 2px solid #4a306d;
            padding: 15px;
            position: relative;
            transition: all 0.2s;
        }
        .article-item:hover {
            border-color: #ffd700;
            box-shadow: 0 0 10px rgba(255, 215, 0, 0.3);
        }
        .article-item.unread { border-left: 4px solid #ffd700; }
        .article-item.read { opacity: 0.7; }
        
        .unread-badge {
            position: absolute;
            top: -8px;
            right: -8px;
            background: #ffd700;
            color: #2d1b4e;
            width: 24px;
            height: 24px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-weight: bold;
            animation: bounce 1s infinite;
        }
        @keyframes bounce {
            0%, 100% { transform: translateY(0); }
            50% { transform: translateY(-3px); }
        }
        
        .article-header {
            display: flex;
            justify-content: space-between;
            margin-bottom: 8px;
            font-size: 9px;
            color: #967bb6;
        }
        .article-feed { color: #7fff00; }
        .article-title {
            color: #fff;
            font-size: 12px;
            text-decoration: none;
            display: block;
            margin-bottom: 10px;
        }
        .article-title:hover { color: #ffd700; }
        .article-summary {
            color: #b8a0d1;
            font-size: 10px;
            line-height: 1.6;
            margin-bottom: 10px;
        }
        .article-actions { display: flex; gap: 15px; }
        .action-btn {
            color: #967bb6;
            cursor: pointer;
            font-size: 10px;
        }
        .action-btn:hover { color: #ffd700; }
        .action-btn.favorited { color: #ff6b6b; }
        
        /* 广场 */
        .plaza { padding: 20px 0; }
        .plaza-desc { color: #967bb6; margin: 10px 0 30px; }
        .site-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(300px, 1fr));
            gap: 20px;
        }
        .site-card {
            background: #1a0f2e;
            border: 3px solid #967bb6;
            padding: 20px;
            position: relative;
        }
        .new-badge {
            position: absolute;
            top: -10px;
            right: 10px;
            background: #7fff00;
            color: #1a0f2e;
            padding: 5px 10px;
            font-size: 9px;
            animation: blink 0.5s infinite;
        }
        .site-header { display: flex; gap: 10px; margin-bottom: 10px; }
        .site-icon { font-size: 24px; }
        .site-info h3 { color: #ffd700; font-size: 11px; margin-bottom: 5px; }
        .site-category { color: #967bb6; font-size: 9px; }
        .site-desc { color: #b8a0d1; font-size: 10px; margin-bottom: 15px; }
        .site-latest { background: #0f0a1e; padding: 10px; margin-bottom: 15px; }
        .latest-title { color: #7fff00; font-size: 9px; margin-bottom: 5px; }
        .latest-item {
            display: block;
            color: #967bb6;
            font-size: 9px;
            padding: 3px 0;
            text-decoration: none;
            overflow: hidden;
            text-overflow: ellipsis;
            white-space: nowrap;
        }
        .latest-item:hover { color: #ffd700; }
        .latest-empty { color: #4a306d; font-size: 9px; }
        .site-footer { display: flex; justify-content: space-between; align-items: center; }
        .update-time { color: #4a306d; font-size: 9px; }
        
        /* 收藏 */
        .favorites-page { padding: 20px 0; }
        .favorites-page h2 { color: #ff6b6b; margin-bottom: 20px; }
        
        /* 设置 */
        .settings-page { max-width: 600px; padding: 20px 0; }
        .settings-section {
            background: #1a0f2e;
            border: 3px solid #967bb6;
            padding: 20px;
            margin-bottom: 20px;
        }
        .settings-section h3 { color: #ffd700; margin-bottom: 15px; }
        .mail-icon { margin: 0 5px; }
        .mail-icon.subscribed { color: #7fff00; animation: blink 1s infinite; }
        
        /* 管理后台 */
        .admin-page { padding: 20px 0; }
        .admin-tabs { display: flex; gap: 5px; margin-bottom: 20px; }
        .admin-tabs .tab {
            padding: 10px 20px;
            background: #4a306d;
            color: #967bb6;
            text-decoration: none;
        }
        .admin-tabs .tab.active { background: #ffd700; color: #1a0f2e; }
        .admin-section {
            background: #1a0f2e;
            border: 3px solid #967bb6;
            padding: 20px;
            margin-bottom: 20px;
        }
        .admin-section h3 { color: #ffd700; margin-bottom: 15px; }
        .form-row { display: flex; gap: 10px; margin-bottom: 10px; flex-wrap: wrap; }
        .form-row input { flex: 1; min-width: 150px; }
        .data-table { width: 100%; border-collapse: collapse; }
        .data-table th, .data-table td {
            padding: 10px;
            text-align: left;
            border-bottom: 1px solid #4a306d;
        }
        .data-table th { color: #ffd700; }
        .review-list { display: flex; flex-direction: column; gap: 15px; }
        .review-item {
            background: #0f0a1e;
            padding: 15px;
            border-left: 4px solid #ffd700;
        }
        .review-info { margin-bottom: 10px; }
        .review-info span { display: block; color: #967bb6; font-size: 10px; margin-top: 5px; }
        .review-actions { display: flex; gap: 10px; align-items: center; }
        .review-actions input { flex: 1; }
        .empty { color: #4a306d; text-align: center; padding: 40px; }
        
        /* 模态框 */
        .modal {
            display: none;
            position: fixed;
            top: 0; left: 0; right: 0; bottom: 0;
            background: rgba(0,0,0,0.8);
            z-index: 100;
            align-items: center;
            justify-content: center;
        }
        .modal.show { display: flex; }
        .modal-content {
            background: #1a0f2e;
            border: 4px solid #ffd700;
            padding: 30px;
            min-width: 400px;
            max-width: 90%;
        }
        .modal-content h3 { color: #ffd700; margin-bottom: 20px; }
        .modal-actions { display: flex; justify-content: flex-end; gap: 10px; margin-top: 20px; }
        
        /* 空状态 */
        .empty-state {
            text-align: center;
            padding: 60px 20px;
            color: #967bb6;
        }
        .empty-icon { font-size: 48px; margin-bottom: 20px; }
        .empty-state .hint { margin-top: 10px; font-size: 10px; }
        
        /* 底部提示栏 */
        .hint-bar {
            position: fixed;
            bottom: 0;
            left: 0;
            right: 0;
            background: #1a0f2e;
            border-top: 4px solid #967bb6;
            padding: 10px 20px;
            font-size: 9px;
            color: #967bb6;
            text-align: center;
        }
        
        /* Toast */
        .toast {
            position: fixed;
            top: 20px;
            right: 20px;
            padding: 15px 20px;
            background: #1a0f2e;
            border: 3px solid;
            z-index: 200;
            animation: slideIn 0.3s;
        }
        @keyframes slideIn {
            from { transform: translateX(100%); opacity: 0; }
            to { transform: translateX(0); opacity: 1; }
        }
        .toast.success { border-color: #7fff00; color: #7fff00; }
        .toast.error { border-color: #ff6b6b; color: #ff6b6b; }
        
        /* 响应式 */
        @media (max-width: 900px) {
            .dashboard { flex-direction: column; }
            .sidebar { width: 100%; }
            .site-grid { grid-template-columns: 1fr; }
        }
    </style>
</head>
<body>
    <?php if ($page !== 'login'): ?>
    <div class="status-bar">
        <div class="status-left">
            <span class="status-item">LV.<span><?php echo is_logged_in() ? 'USER' : 'GUEST'; ?></span></span>
            <?php if (is_logged_in()): ?>
            <span class="status-item">FEEDS:<span><?php echo $stats['feeds']; ?></span></span>
            <span class="status-item">UNREAD:<span><?php echo $stats['unread']; ?></span></span>
            <span class="status-item status-mail <?php echo ($user && $user['subscribe_flag']) ? 'subscribed' : ''; ?>">
                MAIL:<?php echo ($user && $user['subscribe_flag']) ? '[@] ON' : '[@]+[X] OFF'; ?>
            </span>
            <?php endif; ?>
        </div>
        <div class="status-right">
            <?php if (is_logged_in()): ?>
            <span><?php echo e($_SESSION['username']); ?> | </span>
            <a href="#" onclick="logout();return false;" style="color:#ff6b6b;">退出</a>
            <?php else: ?>
            <a href="?page=login" style="color:#7fff00;">登录</a>
            <?php endif; ?>
        </div>
    </div>
    
    <nav class="nav-bar">
        <a href="?page=dashboard" class="nav-item <?php echo $page=='dashboard'?'active':''; ?>">[N] 阅读</a>
        <a href="?page=plaza" class="nav-item <?php echo $page=='plaza'?'active':''; ?>">[W] 广场</a>
        <?php if (is_logged_in()): ?>
        <a href="?page=favorites" class="nav-item <?php echo $page=='favorites'?'active':''; ?>"><3 收藏</a>
        <a href="?page=settings" class="nav-item <?php echo $page=='settings'?'active':''; ?>">[S] 设置</a>
        <?php if (is_admin()): ?>
        <a href="?page=admin" class="nav-item <?php echo $page=='admin'?'active':''; ?>">[A] 管理</a>
        <?php endif; ?>
        <?php endif; ?>
    </nav>
    <?php endif; ?>
    
    <div class="container">
        <?php echo get_page_content($page); ?>
    </div>
    
    <?php get_modals(); ?>
    
    <div class="hint-bar">
        <?php if (is_logged_in()): ?>
        Z:确认 X:取消 M:邮件设置 P:广场 | RSS Quest v<?php echo VERSION; ?>
        <?php else: ?>
        ENTER:登录/注册 | RSS Quest v<?php echo VERSION; ?>
        <?php endif; ?>
    </div>
    
    <script>
    // 工具函数
    function showToast(msg, type = 'success') {
        const toast = document.createElement('div');
        toast.className = 'toast ' + type;
        toast.textContent = type === 'success' ? '[OK] ' + msg : '[NO] ' + msg;
        document.body.appendChild(toast);
        setTimeout(() => toast.remove(), 3000);
    }
    
    function api(action, data = {}, method = 'POST') {
        const url = method === 'GET' ? `?action=${action}&` + new URLSearchParams(data) : `?action=${action}`;
        const options = {
            method: method,
            headers: {'Content-Type': 'application/x-www-form-urlencoded'}
        };
        if (method === 'POST') {
            options.body = new URLSearchParams(data);
        }
        return fetch(url, options).then(r => r.json());
    }
    
    // 登录/注册
    function switchTab(tab) {
        document.querySelectorAll('.auth-tabs .tab').forEach(t => t.classList.remove('active'));
        event.target.classList.add('active');
        document.getElementById('loginForm').style.display = tab === 'login' ? 'block' : 'none';
        document.getElementById('registerForm').style.display = tab === 'register' ? 'block' : 'none';
    }
    
    function doLogin() {
        api('login', {
            username: document.getElementById('loginUser').value,
            password: document.getElementById('loginPass').value,
            captcha: document.getElementById('loginCaptcha').value
        }).then(data => {
            if (data.success) {
                location.href = '?page=dashboard';
            } else {
                showToast(data.error, 'error');
            }
        });
    }
    
    function doRegister() {
        api('register', {
            username: document.getElementById('regUser').value,
            password: document.getElementById('regPass').value,
            email: document.getElementById('regEmail').value,
            captcha: document.getElementById('regCaptcha').value
        }).then(data => {
            if (data.success) {
                showToast('注册成功！');
                setTimeout(() => location.href = '?page=dashboard', 1000);
            } else {
                showToast(data.error, 'error');
            }
        });
    }
    
    function logout() {
        api('logout', {}).then(() => location.href = '?page=login');
    }
    
    // 模态框
    function showModal(id) { document.getElementById(id).classList.add('show'); }
    function closeModal(id) { document.getElementById(id).classList.remove('show'); }
    
    function showAddFeed() { showModal('modalAddFeed'); }
    function showAddCategory() { showModal('modalAddCategory'); }
    
    // Feed操作
    function submitAddFeed() {
        api('add_feed', {
            url: document.getElementById('newFeedUrl').value,
            category_id: document.getElementById('newFeedCategory').value
        }).then(data => {
            if (data.success) {
                showToast('添加成功！');
                location.reload();
            } else {
                showToast(data.error, 'error');
            }
        });
    }
    
    function deleteFeed(id) {
        if (!confirm('确定删除这个RSS源？')) return;
        api('delete_feed', {feed_id: id}).then(() => location.reload());
    }
    
    function refreshFeed(id) {
        // 显示加载状态
        const btn = event.target;
        const originalText = btn.textContent;
        btn.textContent = '[R]更新中...';
        btn.style.opacity = '0.6';
        btn.disabled = true;
        
        // 使用AbortController实现超时
        const controller = new AbortController();
        const timeoutId = setTimeout(() => controller.abort(), 60000); // 60秒超时
        
        fetch('?action=refresh_feed', {
            method: 'POST',
            headers: {'Content-Type': 'application/x-www-form-urlencoded'},
            body: new URLSearchParams({feed_id: id}),
            signal: controller.signal
        })
        .then(r => r.json())
        .then(data => {
            clearTimeout(timeoutId);
            btn.textContent = originalText;
            btn.style.opacity = '1';
            btn.disabled = false;
            
            if (data.success) {
                showToast(`更新完成，新增${data.new_count}篇文章`);
                location.reload();
            } else {
                showToast(data.error || '更新失败', 'error');
            }
        })
        .catch(err => {
            clearTimeout(timeoutId);
            btn.textContent = originalText;
            btn.style.opacity = '1';
            btn.disabled = false;
            
            if (err.name === 'AbortError') {
                showToast('请求超时，请稍后重试', 'error');
            } else {
                showToast('网络错误: ' + err.message, 'error');
            }
        });
    }
    
    // 分类操作
    function submitAddCategory() {
        api('admin_add_category', {
            name: document.getElementById('newCategoryName').value,
            color: document.getElementById('newCategoryColor').value
        }).then(data => {
            if (data.success) {
                showToast('分类创建成功！');
                location.reload();
            } else {
                showToast(data.error, 'error');
            }
        });
    }
    
    function deleteCategory(id) {
        if (!confirm('确定删除这个分类？其中的Feed将变为未分类。')) return;
        api('admin_delete_category', {category_id: id}).then(() => location.reload());
    }
    
    // 文章操作
    function markRead(id, reload = false) {
        api('mark_read', {article_id: id, is_read: 1});
        if (reload) location.reload();
    }
    
    function markAllRead(feedId) {
        api('mark_all_read', {feed_id: feedId}, 'GET').then(() => location.reload());
    }
    
    function toggleFavorite(id, el) {
        api('toggle_favorite', {article_id: id}).then(data => {
            if (data.favorited) {
                el.innerHTML = '<3 已收藏';
                el.classList.add('favorited');
                showToast('已收藏');
            } else {
                el.innerHTML = '</3 收藏';
                el.classList.remove('favorited');
                showToast('已取消收藏');
            }
        });
    }
    
    function toggleUnread(cb) {
        const url = new URL(location.href);
        if (cb.checked) {
            url.searchParams.set('unread', '1');
        } else {
            url.searchParams.delete('unread');
        }
        location.href = url.toString();
    }
    
    // 广场
    function addFromPlaza(plazaId) {
        api('add_from_plaza', {plaza_id: plazaId}).then(data => {
            if (data.success) {
                showToast('已添加到自选！');
                event.target.textContent = '[OK] 已添加';
                event.target.disabled = true;
                event.target.classList.add('disabled');
            } else {
                showToast(data.error, 'error');
            }
        });
    }
    
    // 设置
    function saveSettings() {
        api('update_settings', {
            email: document.getElementById('settingEmail').value,
            subscribe: document.getElementById('settingSubscribe').checked ? 1 : 0
        }).then(data => {
            if (data.success) {
                showToast('设置已保存');
            } else {
                showToast(data.error, 'error');
            }
        });
    }
    
    function changePassword() {
        api('change_password', {
            old_password: document.getElementById('oldPassword').value,
            new_password: document.getElementById('newPassword').value
        }).then(data => {
            if (data.success) {
                showToast('密码已修改');
                document.getElementById('oldPassword').value = '';
                document.getElementById('newPassword').value = '';
            } else {
                showToast(data.error, 'error');
            }
        });
    }
    
    // 管理功能
    function addPlazaSite() {
        api('admin_add_plaza', {
            title: document.getElementById('plazaTitle').value,
            description: document.getElementById('plazaDesc').value,
            feed_url: document.getElementById('plazaUrl').value,
            site_url: document.getElementById('plazaSiteUrl').value,
            category: document.getElementById('plazaCategory').value
        }).then(data => {
            if (data.success) {
                showToast('站点已添加到广场');
                location.reload();
            } else {
                showToast(data.error, 'error');
            }
        });
    }
    
    function deletePlaza(id) {
        if (!confirm('确定从广场删除这个站点？')) return;
        api('admin_delete_plaza', {plaza_id: id}).then(() => location.reload());
    }
    
    function handleRecommend(id, status) {
        api('admin_handle_recommend', {
            recommend_id: id,
            status: status,
            reply: document.getElementById('reply_' + id)?.value || ''
        }).then(data => {
            if (data.success) {
                showToast(status === 1 ? '已通过' : '已拒绝');
                location.reload();
            } else {
                showToast(data.error, 'error');
            }
        });
    }
    
    // 键盘快捷键
    document.addEventListener('keydown', (e) => {
        if (e.key === 'Escape') {
            document.querySelectorAll('.modal').forEach(m => m.classList.remove('show'));
        }
    });
    </script>
</body>
</html>
