<?php

require_once __DIR__ . '/includes/auth.php';
require_once __DIR__ . '/models/Category.php';
require_once __DIR__ . '/models/Link.php';

$categories = Category::all(true);
$errors = [];
$success = false;

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $categoryId = (int) ($_POST['category_id'] ?? 0);
    $title = trim($_POST['title'] ?? '');
    $url = trim($_POST['url'] ?? '');
    $description = trim($_POST['description'] ?? '');

    if ($categoryId <= 0) {
        $errors[] = '请选择分类';
    }
    if (empty($title)) {
        $errors[] = '请输入网站名称';
    }
    if (empty($url)) {
        $errors[] = '请输入网站地址';
    } elseif (!validateUrl($url)) {
        $errors[] = '请输入有效的网站地址';
    }

    if (empty($errors)) {
        $status = Auth::isAdmin() ? 'approved' : 'pending';
        $submittedBy = Auth::id();

        Link::create([
            'category_id' => $categoryId,
            'title' => $title,
            'url' => $url,
            'description' => $description,
            'status' => $status,
            'submitted_by' => $submittedBy
        ]);

        $success = true;
    }
}

$currentUser = Auth::user();
?>
<!DOCTYPE html>
<html lang="zh-CN">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>提交网站 - <?= SITE_NAME ?></title>
    <link rel="stylesheet" href="https://img.penlife.cn/font/OPPO%20Sans%204/result.css">
    <link rel="stylesheet" href="/assets/css/style.css">
</head>
<body>
    <header class="site-header">
        <div class="container">
            <div class="header-content">
                <a href="/" class="site-logo">
                    <span class="logo-icon">🔗</span>
                    <span class="logo-text"><?= SITE_NAME ?></span>
                </a>

                <div class="header-actions">
                    <?php if ($currentUser): ?>
                        <?php if ($currentUser['role'] === 'admin'): ?>
                            <a href="/admin/" class="btn btn-secondary btn-sm">管理后台</a>
                        <?php endif; ?>
                        <a href="/logout.php" class="btn btn-secondary btn-sm">退出</a>
                    <?php else: ?>
                        <a href="/login.php" class="btn btn-secondary btn-sm">登录</a>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </header>

    <main class="main-content">
        <div class="container">
            <div class="submit-page">
                <div class="submit-header">
                    <h1>提交网站</h1>
                    <p>提交您认为有价值的网站，我们会尽快审核</p>
                </div>

                <?php if ($success): ?>
                    <div class="alert alert-success">
                        <?php if (Auth::isAdmin()): ?>
                            网站已成功添加！
                        <?php else: ?>
                            提交成功！您的网站正在等待审核，审核通过后将显示在首页。
                        <?php endif; ?>
                        <div style="margin-top: 1rem;">
                            <a href="/" class="btn btn-primary">返回首页</a>
                            <a href="/submit.php" class="btn btn-secondary">继续提交</a>
                        </div>
                    </div>
                <?php else: ?>
                    <?php if (!empty($errors)): ?>
                        <div class="alert alert-error">
                            <ul>
                                <?php foreach ($errors as $error): ?>
                                    <li><?= h($error) ?></li>
                                <?php endforeach; ?>
                            </ul>
                        </div>
                    <?php endif; ?>

                    <form method="POST" class="submit-form">
                        <div class="form-group">
                            <label for="category_id">分类 *</label>
                            <select id="category_id" name="category_id" required>
                                <option value="">选择分类</option>
                                <?php foreach ($categories as $category): ?>
                                    <option value="<?= $category['id'] ?>">
                                        <?= h($category['name']) ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>

                        <div class="form-group">
                            <label for="title">网站名称 *</label>
                            <input type="text" id="title" name="title" placeholder="例如：GitHub" required>
                        </div>

                        <div class="form-group">
                            <label for="url">网站地址 *</label>
                            <input type="url" id="url" name="url" placeholder="https://example.com" required>
                        </div>

                        <div class="form-group">
                            <label for="description">网站描述</label>
                            <textarea id="description" name="description" rows="3" placeholder="简单描述这个网站的内容或特点"></textarea>
                        </div>

                        <div class="form-actions">
                            <button type="submit" class="btn btn-primary btn-lg">提交</button>
                            <a href="/" class="btn btn-secondary btn-lg">取消</a>
                        </div>

                        <?php if (!Auth::check()): ?>
                            <p class="form-note">
                                提示：<a href="/login.php">登录</a>后提交可获得更好的体验，管理员提交的网站会自动通过审核。
                            </p>
                        <?php elseif (!Auth::isAdmin()): ?>
                            <p class="form-note">
                                提示：您提交的网站需要管理员审核后才会显示。
                            </p>
                        <?php endif; ?>
                    </form>
                <?php endif; ?>
            </div>
        </div>
    </main>

    <footer class="site-footer">
        <div class="container">
            <p>&copy; <?= date('Y') ?> <?= SITE_NAME ?></p>
        </div>
    </footer>
</body>
</html>
