<?php

require_once __DIR__ . '/../config/database.php';

class Link {
    public static function all(string $status = null): array {
        $sql = 'SELECT l.*, c.name as category_name 
                FROM links l 
                LEFT JOIN categories c ON l.category_id = c.id';

        if ($status) {
            $sql .= ' WHERE l.status = :status';
            return db()->fetchAll($sql, ['status' => $status]);
        }

        return db()->fetchAll($sql);
    }

    public static function find(int $id): ?array {
        return db()->fetch(
            'SELECT l.*, c.name as category_name 
             FROM links l 
             LEFT JOIN categories c ON l.category_id = c.id 
             WHERE l.id = :id',
            ['id' => $id]
        );
    }

    public static function create(array $data): int {
        return db()->insert('links', [
            'category_id' => $data['category_id'],
            'title' => $data['title'],
            'url' => $data['url'],
            'description' => $data['description'] ?? null,
            'icon' => $data['icon'] ?? null,
            'sort_order' => $data['sort_order'] ?? 0,
            'status' => $data['status'] ?? 'approved',
            'submitted_by' => $data['submitted_by'] ?? null
        ]);
    }

    public static function update(int $id, array $data): bool {
        $updateData = [];

        $allowedFields = ['category_id', 'title', 'url', 'description', 'icon', 'sort_order', 'status'];

        foreach ($allowedFields as $field) {
            if (isset($data[$field])) {
                $updateData[$field] = $data[$field];
            }
        }

        if (empty($updateData)) {
            return false;
        }

        return db()->update('links', $updateData, 'id = :id', ['id' => $id]) > 0;
    }

    public static function delete(int $id): bool {
        return db()->delete('links', 'id = :id', ['id' => $id]) > 0;
    }

    public static function getByCategory(int $categoryId, string $status = 'approved'): array {
        $sql = 'SELECT * FROM links 
                WHERE category_id = :category_id AND status = :status
                ORDER BY sort_order ASC, id ASC';

        return db()->fetchAll($sql, [
            'category_id' => $categoryId,
            'status' => $status
        ]);
    }

    public static function getApproved(): array {
        $sql = 'SELECT l.*, c.name as category_name, c.slug as category_slug
                FROM links l
                LEFT JOIN categories c ON l.category_id = c.id
                WHERE l.status = "approved" AND c.is_active = 1
                ORDER BY c.sort_order ASC, l.sort_order ASC, l.id ASC';

        return db()->fetchAll($sql);
    }

    public static function getByStatus(string $status): array {
        $sql = 'SELECT l.*, c.name as category_name, u.username as submitter_name
                FROM links l
                LEFT JOIN categories c ON l.category_id = c.id
                LEFT JOIN users u ON l.submitted_by = u.id
                WHERE l.status = :status
                ORDER BY l.created_at DESC';

        return db()->fetchAll($sql, ['status' => $status]);
    }

    public static function search(string $keyword): array {
        $keyword = '%' . $keyword . '%';

        $sql = 'SELECT l.*, c.name as category_name
                FROM links l
                LEFT JOIN categories c ON l.category_id = c.id
                WHERE l.status = "approved" 
                AND (l.title LIKE :keyword OR l.description LIKE :keyword OR l.url LIKE :keyword)
                ORDER BY l.click_count DESC, l.id ASC';

        return db()->fetchAll($sql, ['keyword' => $keyword]);
    }

    public static function incrementClick(int $id): void {
        db()->query(
            'UPDATE links SET click_count = click_count + 1 WHERE id = :id',
            ['id' => $id]
        );
    }

    public static function approve(int $id): bool {
        return self::update($id, ['status' => 'approved']);
    }

    public static function reject(int $id): bool {
        return self::update($id, ['status' => 'rejected']);
    }

    public static function getPendingCount(): int {
        $result = db()->fetch(
            'SELECT COUNT(*) as count FROM links WHERE status = "pending"'
        );
        return (int) $result['count'];
    }

    public static function getStats(): array {
        $stats = db()->fetch(
            'SELECT 
                COUNT(*) as total,
                SUM(CASE WHEN status = "approved" THEN 1 ELSE 0 END) as approved,
                SUM(CASE WHEN status = "pending" THEN 1 ELSE 0 END) as pending,
                SUM(CASE WHEN status = "rejected" THEN 1 ELSE 0 END) as rejected,
                SUM(click_count) as total_clicks
            FROM links'
        );

        return $stats;
    }
}
