<?php

require_once __DIR__ . '/../config/database.php';

class Category {
    public static function all(bool $activeOnly = false): array {
        $sql = 'SELECT * FROM categories';
        if ($activeOnly) {
            $sql .= ' WHERE is_active = 1';
        }
        $sql .= ' ORDER BY sort_order ASC, id ASC';

        return db()->fetchAll($sql);
    }

    public static function find(int $id): ?array {
        return db()->fetch(
            'SELECT * FROM categories WHERE id = :id',
            ['id' => $id]
        );
    }

    public static function findBySlug(string $slug): ?array {
        return db()->fetch(
            'SELECT * FROM categories WHERE slug = :slug',
            ['slug' => $slug]
        );
    }

    public static function create(array $data): int {
        if (empty($data['slug'])) {
            $data['slug'] = generateSlug($data['name']);
        }

        $existing = self::findBySlug($data['slug']);
        if ($existing) {
            $data['slug'] = $data['slug'] . '-' . time();
        }

        return db()->insert('categories', [
            'name' => $data['name'],
            'slug' => $data['slug'],
            'description' => $data['description'] ?? null,
            'icon' => $data['icon'] ?? null,
            'sort_order' => $data['sort_order'] ?? 0,
            'is_active' => $data['is_active'] ?? 1
        ]);
    }

    public static function update(int $id, array $data): bool {
        $updateData = [];

        if (isset($data['name'])) {
            $updateData['name'] = $data['name'];
        }
        if (isset($data['slug'])) {
            $updateData['slug'] = $data['slug'];
        }
        if (isset($data['description'])) {
            $updateData['description'] = $data['description'];
        }
        if (isset($data['icon'])) {
            $updateData['icon'] = $data['icon'];
        }
        if (isset($data['sort_order'])) {
            $updateData['sort_order'] = $data['sort_order'];
        }
        if (isset($data['is_active'])) {
            $updateData['is_active'] = $data['is_active'];
        }

        if (empty($updateData)) {
            return false;
        }

        return db()->update('categories', $updateData, 'id = :id', ['id' => $id]) > 0;
    }

    public static function delete(int $id): bool {
        return db()->delete('categories', 'id = :id', ['id' => $id]) > 0;
    }

    public static function getWithLinkCount(): array {
        $sql = 'SELECT c.*, COUNT(l.id) as link_count 
                FROM categories c 
                LEFT JOIN links l ON c.id = l.category_id AND l.status = "approved"
                GROUP BY c.id 
                ORDER BY c.sort_order ASC, c.id ASC';

        return db()->fetchAll($sql);
    }

    public static function toggleActive(int $id): bool {
        $category = self::find($id);
        if (!$category) {
            return false;
        }

        return self::update($id, ['is_active' => $category['is_active'] ? 0 : 1]);
    }
}
