<?php

require_once __DIR__ . '/includes/auth.php';
require_once __DIR__ . '/models/Category.php';
require_once __DIR__ . '/models/Link.php';

$searchKeyword = trim($_GET['q'] ?? '');

if ($searchKeyword) {
    $links = Link::search($searchKeyword);
    $categories = [];
} else {
    $links = Link::getApproved();
    $categories = Category::all(true);
}

$linksByCategory = [];
foreach ($links as $link) {
    $catId = $link['category_id'];
    if (!isset($linksByCategory[$catId])) {
        $linksByCategory[$catId] = [
            'name' => $link['category_name'],
            'slug' => $link['category_slug'],
            'links' => []
        ];
    }
    $linksByCategory[$catId]['links'][] = $link;
}

$currentUser = Auth::user();
?>
<!DOCTYPE html>
<html lang="zh-CN">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= $searchKeyword ? "搜索: {$searchKeyword} - " : '' ?><?= SITE_NAME ?></title>
    <meta name="description" content="简洁高效的网址导航，收录优质网站资源">
    <link rel="stylesheet" href="https://img.penlife.cn/font/OPPO%20Sans%204/result.css">
    <link rel="stylesheet" href="/assets/css/style.css">
</head>
<body>
    <header class="site-header">
        <div class="container">
            <div class="header-content">
                <a href="/" class="site-logo">
                    <span class="logo-icon">🔗</span>
                    <span class="logo-text"><?= SITE_NAME ?></span>
                </a>

                <form class="search-form" method="GET" action="/">
                    <input type="text" name="q" placeholder="搜索网站..." value="<?= h($searchKeyword) ?>">
                    <button type="submit">
                        <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <circle cx="11" cy="11" r="8"></circle>
                            <line x1="21" y1="21" x2="16.65" y2="16.65"></line>
                        </svg>
                    </button>
                </form>

                <div class="header-actions">
                    <a href="/submit.php" class="btn btn-primary btn-sm">提交网站</a>
                    <?php if ($currentUser): ?>
                        <?php if ($currentUser['role'] === 'admin'): ?>
                            <a href="/admin/" class="btn btn-secondary btn-sm">管理后台</a>
                        <?php endif; ?>
                        <a href="/logout.php" class="btn btn-secondary btn-sm">退出</a>
                    <?php else: ?>
                        <a href="/login.php" class="btn btn-secondary btn-sm">登录</a>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </header>

    <?php if ($searchKeyword): ?>
        <div class="search-result-header">
            <div class="container">
                <h1>搜索结果</h1>
                <p>找到 <?= count($links) ?> 个与 "<?= h($searchKeyword) ?>" 相关的网站</p>
                <a href="/" class="btn btn-secondary btn-sm">清除搜索</a>
            </div>
        </div>
    <?php endif; ?>

    <main class="main-content">
        <div class="container">
            <?php $flash = getFlashMessage(); ?>
            <?php if ($flash): ?>
                <div class="alert alert-<?= h($flash['type']) ?>">
                    <?= h($flash['text']) ?>
                </div>
            <?php endif; ?>

            <?php if (empty($linksByCategory)): ?>
                <div class="empty-state">
                    <div class="empty-icon">🔍</div>
                    <h2>暂无网站</h2>
                    <p><?= $searchKeyword ? '没有找到相关网站，试试其他关键词？' : '还没有收录网站，请稍后再来' ?></p>
                </div>
            <?php else: ?>
                <?php foreach ($linksByCategory as $catId => $category): ?>
                    <section class="category-section" id="category-<?= h($category['slug']) ?>">
                        <div class="category-header">
                            <h2 class="category-title"><?= h($category['name']) ?></h2>
                        </div>
                        <div class="links-grid">
                            <?php foreach ($category['links'] as $link): ?>
                                <a href="/click.php?id=<?= $link['id'] ?>" target="_blank" rel="noopener" class="link-card">
                                    <div class="link-icon-wrapper">
                                        <img src="<?= h($link['icon'] ?: getFaviconUrl($link['url'])) ?>" alt="" class="link-icon" onerror="this.src='/assets/images/default-icon.png'">
                                    </div>
                                    <div class="link-content">
                                        <h3 class="link-title"><?= h($link['title']) ?></h3>
                                        <p class="link-description"><?= h(truncate($link['description'] ?? $link['url'], 80)) ?></p>
                                    </div>
                                </a>
                            <?php endforeach; ?>
                        </div>
                    </section>
                <?php endforeach; ?>
            <?php endif; ?>
        </div>
    </main>

    <footer class="site-footer">
        <div class="container">
            <p>&copy; <?= date('Y') ?> <?= SITE_NAME ?> · 简洁高效的网址导航</p>
        </div>
    </footer>

    <script>
        document.querySelectorAll('.link-card').forEach(card => {
            card.addEventListener('click', function(e) {
                const href = this.getAttribute('href');
                if (href.startsWith('/click.php')) {
                    fetch(href).catch(() => {});
                }
            });
        });
    </script>
</body>
</html>
