<?php

session_name(SESSION_NAME);
session_start();

require_once __DIR__ . '/functions.php';

class Auth {
    public static function login(array $user): void {
        $_SESSION['user_id'] = $user['id'];
        $_SESSION['username'] = $user['username'];
        $_SESSION['user_role'] = $user['role'];
        session_regenerate_id(true);
    }

    public static function logout(): void {
        $_SESSION = [];
        if (ini_get('session.use_cookies')) {
            $params = session_get_cookie_params();
            setcookie(
                session_name(),
                '',
                time() - 42000,
                $params['path'],
                $params['domain'],
                $params['secure'],
                $params['httponly']
            );
        }
        session_destroy();
    }

    public static function check(): bool {
        return isset($_SESSION['user_id']);
    }

    public static function isAdmin(): bool {
        return isset($_SESSION['user_role']) && $_SESSION['user_role'] === 'admin';
    }

    public static function user(): ?array {
        if (!self::check()) {
            return null;
        }
        return [
            'id' => $_SESSION['user_id'],
            'username' => $_SESSION['username'],
            'role' => $_SESSION['user_role']
        ];
    }

    public static function id(): ?int {
        return $_SESSION['user_id'] ?? null;
    }

    public static function requireLogin(): void {
        if (!self::check()) {
            redirect('/login.php');
        }
    }

    public static function requireAdmin(): void {
        self::requireLogin();
        if (!self::isAdmin()) {
            redirect('/');
        }
    }

    public static function hashPassword(string $password): string {
        return password_hash($password, PASSWORD_DEFAULT);
    }

    public static function verifyPassword(string $password, string $hash): bool {
        return password_verify($password, $hash);
    }

    public static function findByUsername(string $username): ?array {
        return db()->fetch(
            'SELECT * FROM users WHERE username = :username',
            ['username' => $username]
        );
    }

    public static function findByEmail(string $email): ?array {
        return db()->fetch(
            'SELECT * FROM users WHERE email = :email',
            ['email' => $email]
        );
    }

    public static function attempt(string $username, string $password): array {
        $user = self::findByUsername($username);

        if (!$user) {
            return ['success' => false, 'message' => '用户名或密码错误'];
        }

        if (!self::verifyPassword($password, $user['password'])) {
            return ['success' => false, 'message' => '用户名或密码错误'];
        }

        self::login($user);
        return ['success' => true, 'message' => '登录成功'];
    }

    public static function register(string $username, string $email, string $password): array {
        if (self::findByUsername($username)) {
            return ['success' => false, 'message' => '用户名已存在'];
        }

        if (self::findByEmail($email)) {
            return ['success' => false, 'message' => '邮箱已被注册'];
        }

        if (strlen($password) < 6) {
            return ['success' => false, 'message' => '密码长度至少6位'];
        }

        $userId = db()->insert('users', [
            'username' => $username,
            'email' => $email,
            'password' => self::hashPassword($password),
            'role' => 'user'
        ]);

        return ['success' => true, 'message' => '注册成功', 'user_id' => $userId];
    }
}
