<?php
declare(strict_types=1);

session_start();

$installedFile = __DIR__ . '/.installed';
if (file_exists($installedFile)) {
    die('系统已安装，如需重新安装请删除 .installed 文件');
}

$step = $_GET['step'] ?? '1';
$error = '';
$success = '';

function testDBConnection(array $config): bool
{
    try {
        $dsn = "mysql:host={$config['host']};port={$config['port']};charset={$config['charset']}";
        $pdo = new PDO($dsn, $config['user'], $config['pass'], [
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
        ]);
        return true;
    } catch (PDOException $e) {
        return false;
    }
}

function createDatabase(PDO $pdo, string $name): bool
{
    try {
        $pdo->exec("CREATE DATABASE IF NOT EXISTS `{$name}` DEFAULT CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci");
        return true;
    } catch (PDOException $e) {
        return false;
    }
}

function createTables(PDO $pdo): bool
{
    $sql = "
        CREATE TABLE IF NOT EXISTS comments (
            id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
            page_id VARCHAR(255) NOT NULL,
            parent_id INT UNSIGNED NULL,
            author VARCHAR(100) NOT NULL DEFAULT '匿名用户',
            email VARCHAR(255) NOT NULL,
            website VARCHAR(255) NULL,
            content TEXT NOT NULL,
            ip VARCHAR(45) NOT NULL,
            user_agent VARCHAR(500) NULL,
            status ENUM('pending', 'approved', 'spam') NOT NULL DEFAULT 'pending',
            created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
            updated_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            INDEX idx_page_id (page_id),
            INDEX idx_parent_id (parent_id),
            INDEX idx_status (status),
            INDEX idx_created_at (created_at)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

        CREATE TABLE IF NOT EXISTS comment_likes (
            id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
            comment_id INT UNSIGNED NOT NULL,
            ip VARCHAR(45) NOT NULL,
            created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
            UNIQUE KEY uk_comment_ip (comment_id, ip),
            INDEX idx_comment_id (comment_id)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

        CREATE TABLE IF NOT EXISTS admins (
            id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
            username VARCHAR(50) NOT NULL UNIQUE,
            password VARCHAR(255) NOT NULL,
            created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

        CREATE TABLE IF NOT EXISTS admin_sessions (
            id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
            admin_id INT UNSIGNED NOT NULL,
            session_key VARCHAR(64) NOT NULL UNIQUE,
            expires_at DATETIME NOT NULL,
            created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
            INDEX idx_session_key (session_key),
            INDEX idx_expires_at (expires_at)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
    ";

    try {
        $pdo->exec($sql);
        return true;
    } catch (PDOException $e) {
        return false;
    }
}

function createAdmin(PDO $pdo, string $username, string $password): bool
{
    $hashedPassword = password_hash($password, PASSWORD_DEFAULT);
    try {
        $stmt = $pdo->prepare('INSERT INTO admins (username, password) VALUES (?, ?)');
        $stmt->execute([$username, $hashedPassword]);
        return true;
    } catch (PDOException $e) {
        return false;
    }
}

function updateConfig(array $newConfig): bool
{
    $configFile = __DIR__ . '/config.php';
    $content = "<?php\ndeclare(strict_types=1);\n\nreturn " . var_export($newConfig, true) . ";\n";
    return file_put_contents($configFile, $content) !== false;
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if ($step === '2') {
        $dbConfig = [
            'host' => $_POST['db_host'] ?? 'localhost',
            'port' => (int)($_POST['db_port'] ?? 3306),
            'name' => $_POST['db_name'] ?? 'comment_system',
            'user' => $_POST['db_user'] ?? '',
            'pass' => $_POST['db_pass'] ?? '',
            'charset' => 'utf8mb4',
        ];

        if (!testDBConnection($dbConfig)) {
            $error = '数据库连接失败，请检查配置';
        } else {
            $_SESSION['db_config'] = $dbConfig;
            header('Location: ?step=3');
            exit;
        }
    } elseif ($step === '3') {
        $dbConfig = $_SESSION['db_config'] ?? null;

        if (!$dbConfig) {
            header('Location: ?step=2');
            exit;
        }

        try {
            $dsn = "mysql:host={$dbConfig['host']};port={$dbConfig['port']};charset={$dbConfig['charset']}";
            $pdo = new PDO($dsn, $dbConfig['user'], $dbConfig['pass'], [
                PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
            ]);

            if (!createDatabase($pdo, $dbConfig['name'])) {
                throw new Exception('创建数据库失败');
            }

            $pdo->exec("USE `{$dbConfig['name']}`");

            if (!createTables($pdo)) {
                throw new Exception('创建数据表失败');
            }

            $username = trim($_POST['admin_user'] ?? '');
            $password = $_POST['admin_pass'] ?? '';
            $passwordConfirm = $_POST['admin_pass_confirm'] ?? '';

            if (strlen($username) < 3) {
                throw new Exception('用户名至少3个字符');
            }

            if (strlen($password) < 6) {
                throw new Exception('密码至少6个字符');
            }

            if ($password !== $passwordConfirm) {
                throw new Exception('两次密码不一致');
            }

            if (!createAdmin($pdo, $username, $password)) {
                throw new Exception('创建管理员失败');
            }

            $currentConfig = require __DIR__ . '/config.php';
            $currentConfig['database'] = $dbConfig;
            $currentConfig['admin']['username'] = $username;

            if (!updateConfig($currentConfig)) {
                throw new Exception('保存配置失败');
            }

            file_put_contents($installedFile, date('Y-m-d H:i:s'));

            header('Location: ?step=4');
            exit;
        } catch (Exception $e) {
            $error = $e->getMessage();
        }
    }
}
?>
<!DOCTYPE html>
<html lang="zh-CN">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>安装向导 - 冰裂纹 Crackle</title>
    <link rel="stylesheet" href="https://img.penlife.cn/font/OPPO%20Sans%204/result.css">
    <script src="https://unpkg.com/@phosphor-icons/web"></script>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: 'OPPO Sans 4.0', -apple-system, BlinkMacSystemFont, sans-serif;
            background: #fff;
            color: #1a1a1a;
            min-height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
            padding: 20px;
        }

        .container {
            width: 100%;
            max-width: 480px;
        }

        .header {
            text-align: center;
            margin-bottom: 40px;
        }

        .header h1 {
            font-size: 28px;
            font-weight: 600;
            margin-bottom: 8px;
        }

        .header h1 small {
            font-size: 16px;
            font-weight: 400;
            color: #666;
            margin-left: 8px;
        }

        .header p {
            color: #666;
            font-size: 14px;
        }

        .steps {
            display: flex;
            justify-content: center;
            gap: 8px;
            margin-bottom: 32px;
        }

        .step-item {
            width: 32px;
            height: 32px;
            border-radius: 50%;
            background: #e1e4e8;
            color: #666;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 14px;
            font-weight: 500;
        }

        .step-item.active {
            background: #0066cc;
            color: #fff;
        }

        .step-item.done {
            background: #28a745;
            color: #fff;
        }

        .step-line {
            width: 40px;
            height: 2px;
            background: #e1e4e8;
            align-self: center;
        }

        .step-line.done {
            background: #28a745;
        }

        .card {
            background: #fff;
            border: 1px solid #e1e4e8;
            border-radius: 8px;
            padding: 24px;
        }

        .card h2 {
            font-size: 18px;
            font-weight: 600;
            margin-bottom: 20px;
        }

        .form-group {
            margin-bottom: 16px;
        }

        .form-group label {
            display: block;
            font-size: 14px;
            font-weight: 500;
            margin-bottom: 6px;
            color: #1a1a1a;
        }

        .form-group input {
            width: 100%;
            padding: 10px 12px;
            border: 1px solid #e1e4e8;
            border-radius: 6px;
            font-size: 14px;
            font-family: inherit;
            transition: border-color 0.2s;
        }

        .form-group input:focus {
            outline: none;
            border-color: #0066cc;
        }

        .form-row {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 12px;
        }

        .btn {
            width: 100%;
            padding: 12px;
            background: #0066cc;
            color: #fff;
            border: none;
            border-radius: 6px;
            font-size: 14px;
            font-weight: 500;
            font-family: inherit;
            cursor: pointer;
            transition: background 0.2s;
        }

        .btn:hover {
            background: #0052a3;
        }

        .error {
            background: #fff5f5;
            border: 1px solid #feb2b2;
            color: #c53030;
            padding: 12px;
            border-radius: 6px;
            font-size: 14px;
            margin-bottom: 16px;
        }

        .success-box {
            text-align: center;
            padding: 20px 0;
        }

        .success-box i {
            font-size: 48px;
            color: #28a745;
            margin-bottom: 16px;
        }

        .success-box h3 {
            font-size: 20px;
            margin-bottom: 8px;
        }

        .success-box p {
            color: #666;
            font-size: 14px;
            margin-bottom: 24px;
        }

        .success-box .btn {
            display: inline-block;
            width: auto;
            padding: 12px 32px;
            text-decoration: none;
        }

        .requirements {
            margin-bottom: 20px;
        }

        .req-item {
            display: flex;
            align-items: center;
            gap: 8px;
            padding: 8px 0;
            font-size: 14px;
        }

        .req-item i {
            font-size: 18px;
        }

        .req-item.pass i {
            color: #28a745;
        }

        .req-item.fail i {
            color: #d73a49;
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1>冰裂纹 <small>Crackle</small></h1>
            <p>安装向导</p>
        </div>

        <div class="steps">
            <div class="step-item <?= $step >= 1 ? ($step > 1 ? 'done' : 'active') : '' ?>">1</div>
            <div class="step-line <?= $step > 1 ? 'done' : '' ?>"></div>
            <div class="step-item <?= $step >= 2 ? ($step > 2 ? 'done' : 'active') : '' ?>">2</div>
            <div class="step-line <?= $step > 2 ? 'done' : '' ?>"></div>
            <div class="step-item <?= $step >= 3 ? ($step > 3 ? 'done' : 'active') : '' ?>">3</div>
            <div class="step-line <?= $step > 3 ? 'done' : '' ?>"></div>
            <div class="step-item <?= $step >= 4 ? 'active' : '' ?>">4</div>
        </div>

        <div class="card">
            <?php if ($step === '1'): ?>
                <h2>环境检查</h2>
                <div class="requirements">
                    <?php
                    $requirements = [
                        'PHP >= 8.0' => version_compare(PHP_VERSION, '8.0.0', '>='),
                        'PDO 扩展' => extension_loaded('pdo'),
                        'PDO MySQL 扩展' => extension_loaded('pdo_mysql'),
                        'JSON 扩展' => extension_loaded('json'),
                        'Mbstring 扩展' => extension_loaded('mbstring'),
                        'config.php 可写' => is_writable(__DIR__ . '/config.php'),
                    ];
                    $allPass = true;
                    foreach ($requirements as $name => $pass):
                        if (!$pass) $allPass = false;
                    ?>
                        <div class="req-item <?= $pass ? 'pass' : 'fail' ?>">
                            <i class="ph-bold <?= $pass ? 'ph-check-circle' : 'ph-x-circle' ?>"></i>
                            <span><?= $name ?></span>
                        </div>
                    <?php endforeach; ?>
                </div>
                <?php if ($allPass): ?>
                    <a href="?step=2" class="btn">下一步</a>
                <?php else: ?>
                    <button class="btn" disabled style="opacity: 0.5; cursor: not-allowed;">环境不满足要求</button>
                <?php endif; ?>

            <?php elseif ($step === '2'): ?>
                <h2>数据库配置</h2>
                <?php if ($error): ?>
                    <div class="error"><?= htmlspecialchars($error) ?></div>
                <?php endif; ?>
                <form method="post">
                    <div class="form-row">
                        <div class="form-group">
                            <label>主机地址</label>
                            <input type="text" name="db_host" value="localhost" required>
                        </div>
                        <div class="form-group">
                            <label>端口</label>
                            <input type="number" name="db_port" value="3306" required>
                        </div>
                    </div>
                    <div class="form-group">
                        <label>数据库名</label>
                        <input type="text" name="db_name" value="comment_system" required>
                    </div>
                    <div class="form-group">
                        <label>用户名</label>
                        <input type="text" name="db_user" required>
                    </div>
                    <div class="form-group">
                        <label>密码</label>
                        <input type="password" name="db_pass">
                    </div>
                    <button type="submit" class="btn">测试连接</button>
                </form>

            <?php elseif ($step === '3'): ?>
                <h2>管理员账户</h2>
                <?php if ($error): ?>
                    <div class="error"><?= htmlspecialchars($error) ?></div>
                <?php endif; ?>
                <form method="post">
                    <div class="form-group">
                        <label>用户名</label>
                        <input type="text" name="admin_user" minlength="3" required>
                    </div>
                    <div class="form-group">
                        <label>密码</label>
                        <input type="password" name="admin_pass" minlength="6" required>
                    </div>
                    <div class="form-group">
                        <label>确认密码</label>
                        <input type="password" name="admin_pass_confirm" required>
                    </div>
                    <button type="submit" class="btn">完成安装</button>
                </form>

            <?php elseif ($step === '4'): ?>
                <div class="success-box">
                    <i class="ph-bold ph-check-circle"></i>
                    <h3>安装成功</h3>
                    <p>冰裂纹已安装完成，请删除 install.php 文件后访问管理后台</p>
                    <a href="admin.php" class="btn">进入后台</a>
                </div>
            <?php endif; ?>
        </div>
    </div>
</body>
</html>
