(function() {
    const FONT_URL = 'https://img.penlife.cn/font/OPPO%20Sans%204/result.css';

    function loadStyles(apiBase) {
        if (!document.querySelector(`link[href="${FONT_URL}"]`)) {
            const fontLink = document.createElement('link');
            fontLink.rel = 'stylesheet';
            fontLink.href = FONT_URL;
            document.head.appendChild(fontLink);
        }

        const iconsUrl = `${apiBase}?route=/proxy/icons&file=bold`;
        if (!document.querySelector(`link[href="${iconsUrl}"]`)) {
            const iconsLink = document.createElement('link');
            iconsLink.rel = 'stylesheet';
            iconsLink.href = iconsUrl;
            document.head.appendChild(iconsLink);
        }
    }

    function detectApiBase() {
        if (window.COMMENT_API_BASE) {
            return window.COMMENT_API_BASE.replace(/\/$/, '');
        }
        
        const script = document.currentScript;
        if (script) {
            if (script.dataset.apiBase) {
                return script.dataset.apiBase.replace(/\/$/, '');
            }
            if (script.src) {
                const src = new URL(script.src, window.location.href);
                const path = src.pathname.replace(/[^/]+$/, '') + 'api.php';
                return src.origin + path;
            }
        }
        
        const currentPath = window.location.pathname;
        const parts = currentPath.split('/');
        for (let i = parts.length; i >= 0; i--) {
            const testPath = parts.slice(0, i).join('/') + '/api.php';
            if (testPath) {
                return window.location.origin + testPath;
            }
        }
        return window.location.origin + '/api.php';
    }

    class CommentWidget {
        constructor(options) {
            this.options = {
                apiBase: detectApiBase(),
                pageId: window.location.pathname,
                container: null,
                ...options,
            };

            this.container = typeof this.options.container === 'string'
                ? document.querySelector(this.options.container)
                : this.options.container;

            if (!this.container) {
                console.error('CommentWidget: container not found');
                return;
            }

            loadStyles(this.options.apiBase);

            this.comments = [];
            this.replyingTo = null;
            this.stickers = [];
            this.stickerMap = {};
            this.init();
        }

        async init() {
            await this.loadStickers();
            this.render();
            await this.loadComments();
        }

        async loadStickers() {
            try {
                const res = await fetch(`${this.options.apiBase}?route=/proxy/stickers`);
                const data = await res.json();
                const firstKey = Object.keys(data)[0];
                if (data[firstKey] && data[firstKey].container) {
                    this.stickers = data[firstKey].container.map(s => {
                        const nameMatch = s.text.match(/heo-(.+)/);
                        const name = nameMatch ? nameMatch[1] : '';
                        return { name };
                    }).filter(s => s.name);
                    this.stickers.forEach(s => {
                        this.stickerMap[s.name] = `${this.options.apiBase}?route=/proxy/sticker&name=${encodeURIComponent(s.name)}`;
                    });
                }
            } catch (e) {
                console.error('Failed to load stickers:', e);
            }
        }

        async api(endpoint, method = 'GET', data = null) {
            const options = {
                method,
                headers: { 'Content-Type': 'application/json' },
            };
            if (data) options.body = JSON.stringify(data);

            try {
                const url = endpoint.includes('?') 
                    ? `${this.options.apiBase}?route=${endpoint}` 
                    : `${this.options.apiBase}?route=${endpoint}`;
                const res = await fetch(url, options);
                return await res.json();
            } catch (e) {
                console.error('API Error:', e);
                return { error: '网络错误' };
            }
        }

        async loadComments() {
            const result = await this.api(`/comments?page_id=${encodeURIComponent(this.options.pageId)}`);
            if (result.success) {
                this.comments = result.data.comments;
                this.renderComments();
            }
        }

        render() {
            this.container.innerHTML = `
                <div class="comment-widget">
                    <div class="comment-widget-header">
                        <h3>评论</h3>
                    </div>
                    <div class="comment-form">
                        <div class="comment-form-row">
                            <div class="form-group">
                                <input type="text" id="comment-author" placeholder="昵称">
                            </div>
                            <div class="form-group">
                                <input type="email" id="comment-email" placeholder="邮箱" required>
                            </div>
                            <div class="form-group">
                                <input type="url" id="comment-website" placeholder="网站（选填）">
                            </div>
                        </div>
                        <div class="form-group" style="position:relative;">
                            <textarea id="comment-content" placeholder="写下你的评论..." rows="3"></textarea>
                            <button type="button" class="sticker-btn" id="sticker-btn" title="表情">
                                <i class="ph-bold ph-smiley"></i>
                            </button>
                        </div>
                        <div class="sticker-panel" id="sticker-panel" style="display:none;">
                            <div class="sticker-grid">
                                ${this.stickers.map(s => `
                                    <img src="${this.options.apiBase}?route=/proxy/sticker&name=${encodeURIComponent(s.name)}" 
                                         class="sticker-item" 
                                         data-name="${s.name}"
                                         title="${s.name}"
                                         loading="lazy">
                                `).join('')}
                            </div>
                        </div>
                        <div class="comment-form-actions">
                            <button type="button" class="btn btn-primary" id="submit-comment">
                                <i class="ph-bold ph-paper-plane-tilt"></i> 发表评论
                            </button>
                        </div>
                    </div>
                    <div class="comment-list" id="comment-list"></div>
                </div>
            `;

            this.bindFormEvents();
        }

        bindFormEvents() {
            const stickerBtn = this.container.querySelector('#sticker-btn');
            const stickerPanel = this.container.querySelector('#sticker-panel');
            const contentTextarea = this.container.querySelector('#comment-content');

            stickerBtn.addEventListener('click', (e) => {
                e.stopPropagation();
                stickerPanel.style.display = stickerPanel.style.display === 'none' ? 'block' : 'none';
            });

            document.addEventListener('click', (e) => {
                if (!stickerPanel.contains(e.target) && e.target !== stickerBtn) {
                    stickerPanel.style.display = 'none';
                }
            });

            stickerPanel.addEventListener('click', (e) => {
                const stickerItem = e.target.closest('.sticker-item');
                if (stickerItem) {
                    const name = stickerItem.dataset.name;
                    const cursorPos = contentTextarea.selectionStart;
                    const textBefore = contentTextarea.value.substring(0, cursorPos);
                    const textAfter = contentTextarea.value.substring(cursorPos);
                    contentTextarea.value = textBefore + `[${name}]` + textAfter;
                    contentTextarea.focus();
                    stickerPanel.style.display = 'none';
                }
            });

            this.container.querySelector('#submit-comment').addEventListener('click', () => this.submitComment());
        }

        parseStickers(text) {
            return text.replace(/\[([^\]]+)\]/g, (match, name) => {
                if (this.stickerMap[name]) {
                    return `<img src="${this.stickerMap[name]}" class="sticker-inline" alt="${name}">`;
                }
                return match;
            });
        }

        renderComments() {
            const list = this.container.querySelector('#comment-list');

            if (this.comments.length === 0) {
                list.innerHTML = `
                    <div class="empty-state">
                        <i class="ph-bold ph-chat-circle-text"></i>
                        <p>暂无评论，来抢沙发吧</p>
                    </div>
                `;
                return;
            }

            list.innerHTML = this.comments.map(comment => this.renderComment(comment)).join('');
            this.bindCommentEvents();
        }

        renderComment(comment) {
            const replies = comment.replies || [];
            const timeAgo = this.timeAgo(comment.created_at);
            const initial = (comment.author || '匿')[0].toUpperCase();
            const content = this.parseStickers(this.escapeHtml(comment.content)).replace(/\n/g, '<br>');

            return `
                <div class="comment-card" data-id="${comment.id}">
                    <div class="comment-card-header">
                        <div class="comment-avatar">${initial}</div>
                        <div class="comment-card-info">
                            <div class="comment-card-name">
                                ${this.escapeHtml(comment.author)}
                                ${comment.website ? `<a href="${this.escapeHtml(comment.website)}" target="_blank" rel="noopener"><i class="ph-bold ph-link"></i></a>` : ''}
                            </div>
                            <div class="comment-card-time">${timeAgo}</div>
                        </div>
                        <div class="comment-card-actions">
                            <button class="comment-card-btn like-btn ${comment.liked ? 'liked' : ''}" data-id="${comment.id}">
                                <i class="ph-bold ph-heart"></i>
                                <span>${comment.like_count || 0}</span>
                            </button>
                            <button class="comment-card-btn reply-btn" data-id="${comment.id}">
                                <i class="ph-bold ph-chat-circle"></i>
                                回复
                            </button>
                        </div>
                    </div>
                    <div class="comment-card-body">${content}</div>
                    ${replies.length > 0 ? `
                        <div class="comment-replies">
                            ${replies.map(reply => this.renderReply(reply)).join('')}
                        </div>
                    ` : ''}
                </div>
            `;
        }

        renderReply(reply) {
            const timeAgo = this.timeAgo(reply.created_at);
            const initial = (reply.author || '匿')[0].toUpperCase();
            const content = this.parseStickers(this.escapeHtml(reply.content)).replace(/\n/g, '<br>');

            return `
                <div class="comment-reply" data-id="${reply.id}">
                    <div class="comment-card-header">
                        <div class="comment-avatar" style="width:28px;height:28px;font-size:12px;">${initial}</div>
                        <div class="comment-card-info">
                            <div class="comment-card-name">${this.escapeHtml(reply.author)}</div>
                            <div class="comment-card-time">${timeAgo}</div>
                        </div>
                        <button class="comment-card-btn like-btn ${reply.liked ? 'liked' : ''}" data-id="${reply.id}">
                            <i class="ph-bold ph-heart"></i>
                            <span>${reply.like_count || 0}</span>
                        </button>
                    </div>
                    <div class="comment-card-body">${content}</div>
                </div>
            `;
        }

        bindCommentEvents() {
            this.container.querySelectorAll('.like-btn').forEach(btn => {
                btn.addEventListener('click', async () => {
                    const id = btn.dataset.id;
                    const result = await this.api(`/comments/${id}/like`, 'POST');
                    if (result.success) {
                        btn.classList.toggle('liked', result.data.liked);
                        btn.querySelector('span').textContent = result.data.count;
                    }
                });
            });

            this.container.querySelectorAll('.reply-btn').forEach(btn => {
                btn.addEventListener('click', () => {
                    const id = btn.dataset.id;
                    this.showReplyForm(id);
                });
            });
        }

        showReplyForm(commentId) {
            const existing = this.container.querySelector('.reply-form');
            if (existing) existing.remove();

            const comment = this.container.querySelector(`.comment-card[data-id="${commentId}"]`);
            if (!comment) return;

            const form = document.createElement('div');
            form.className = 'reply-form';
            form.innerHTML = `
                <div class="form-group">
                    <input type="text" class="reply-author" placeholder="昵称">
                </div>
                <div class="form-group">
                    <input type="email" class="reply-email" placeholder="邮箱" required>
                </div>
                <div class="form-group" style="position:relative;">
                    <textarea class="reply-content" placeholder="写下你的回复..." rows="2"></textarea>
                    <button type="button" class="sticker-btn reply-sticker-btn" title="表情">
                        <i class="ph-bold ph-smiley"></i>
                    </button>
                </div>
                <div class="sticker-panel reply-sticker-panel" style="display:none;">
                    <div class="sticker-grid">
                        ${this.stickers.map(s => `
                            <img src="${this.options.apiBase}?route=/proxy/sticker&name=${encodeURIComponent(s.name)}" 
                                 class="sticker-item" 
                                 data-name="${s.name}"
                                 title="${s.name}"
                                 loading="lazy">
                        `).join('')}
                    </div>
                </div>
                <div class="comment-form-actions">
                    <button type="button" class="btn cancel-reply">取消</button>
                    <button type="button" class="btn btn-primary submit-reply">回复</button>
                </div>
            `;

            comment.querySelector('.comment-card-body').after(form);

            const stickerBtn = form.querySelector('.reply-sticker-btn');
            const stickerPanel = form.querySelector('.reply-sticker-panel');
            const contentTextarea = form.querySelector('.reply-content');

            stickerBtn.addEventListener('click', (e) => {
                e.stopPropagation();
                stickerPanel.style.display = stickerPanel.style.display === 'none' ? 'block' : 'none';
            });

            stickerPanel.addEventListener('click', (e) => {
                const stickerItem = e.target.closest('.sticker-item');
                if (stickerItem) {
                    const name = stickerItem.dataset.name;
                    const cursorPos = contentTextarea.selectionStart;
                    const textBefore = contentTextarea.value.substring(0, cursorPos);
                    const textAfter = contentTextarea.value.substring(cursorPos);
                    contentTextarea.value = textBefore + `[${name}]` + textAfter;
                    contentTextarea.focus();
                    stickerPanel.style.display = 'none';
                }
            });

            form.querySelector('.cancel-reply').addEventListener('click', () => form.remove());
            form.querySelector('.submit-reply').addEventListener('click', async () => {
                const author = form.querySelector('.reply-author').value.trim();
                const email = form.querySelector('.reply-email').value.trim();
                const content = form.querySelector('.reply-content').value.trim();

                if (!email) {
                    alert('请输入邮箱');
                    return;
                }

                if (!content) {
                    alert('请输入回复内容');
                    return;
                }

                const result = await this.api('/comments', 'POST', {
                    page_id: this.options.pageId,
                    parent_id: parseInt(commentId),
                    author: author || '匿名用户',
                    email: email,
                    content,
                });

                if (result.success) {
                    alert(result.message);
                    await this.loadComments();
                } else {
                    alert(result.error || '回复失败');
                }
            });
        }

        async submitComment() {
            const author = this.container.querySelector('#comment-author').value.trim();
            const email = this.container.querySelector('#comment-email').value.trim();
            const website = this.container.querySelector('#comment-website').value.trim();
            const content = this.container.querySelector('#comment-content').value.trim();

            if (!email) {
                alert('请输入邮箱');
                return;
            }

            if (!content) {
                alert('请输入评论内容');
                return;
            }

            const btn = this.container.querySelector('#submit-comment');
            btn.disabled = true;
            btn.textContent = '提交中...';

            const result = await this.api('/comments', 'POST', {
                page_id: this.options.pageId,
                author: author || '匿名用户',
                email: email,
                website: website || null,
                content,
            });

            btn.disabled = false;
            btn.innerHTML = '<i class="ph-bold ph-paper-plane-tilt"></i> 发表评论';

            if (result.success) {
                this.container.querySelector('#comment-content').value = '';
                alert(result.message);
                if (result.data.status === 'approved') {
                    await this.loadComments();
                }
            } else {
                alert(result.error || '评论失败');
            }
        }

        timeAgo(dateStr) {
            const date = new Date(dateStr);
            const now = new Date();
            const seconds = Math.floor((now - date) / 1000);

            const intervals = [
                { label: '年', seconds: 31536000 },
                { label: '个月', seconds: 2592000 },
                { label: '周', seconds: 604800 },
                { label: '天', seconds: 86400 },
                { label: '小时', seconds: 3600 },
                { label: '分钟', seconds: 60 },
            ];

            for (const interval of intervals) {
                const count = Math.floor(seconds / interval.seconds);
                if (count >= 1) {
                    return `${count}${interval.label}前`;
                }
            }

            return '刚刚';
        }

        escapeHtml(str) {
            if (!str) return '';
            const div = document.createElement('div');
            div.textContent = str;
            return div.innerHTML;
        }
    }

    window.CommentWidget = CommentWidget;

    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', autoInit);
    } else {
        autoInit();
    }

    function autoInit() {
        const container = document.querySelector('#comments');
        if (container && !container.dataset.initialized) {
            container.dataset.initialized = 'true';
            new CommentWidget({ container: '#comments' });
        }
    }
})();
