<?php
declare(strict_types=1);

header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

function proxyRequest(string $url, string $contentType, bool $rewriteUrls = false): void
{
    $cacheFile = __DIR__ . '/cache/' . md5($url);
    $cacheDir = dirname($cacheFile);
    
    if (!is_dir($cacheDir)) {
        mkdir($cacheDir, 0755, true);
    }

    if (file_exists($cacheFile) && filemtime($cacheFile) > time() - 86400 * 7) {
        header('Content-Type: ' . $contentType);
        header('Cache-Control: public, max-age=604800');
        readfile($cacheFile);
        exit;
    }

    $ch = curl_init($url);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_TIMEOUT => 30,
        CURLOPT_SSL_VERIFYPEER => true,
        CURLOPT_USERAGENT => 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36',
    ]);

    $data = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    if ($httpCode !== 200 || empty($data)) {
        http_response_code(404);
        exit;
    }

    if ($rewriteUrls) {
        $baseUrl = dirname($url);
        $currentUrl = (isset($_SERVER['HTTPS']) ? 'https' : 'http') . '://' . $_SERVER['HTTP_HOST'] . $_SERVER['REQUEST_URI'];
        $apiBase = strtok($currentUrl, '?');
        
        $data = preg_replace_callback(
            '/url\(\s*[\'"]?([^\'")\s]+)[\'"]?\s*\)/i',
            function($matches) use ($baseUrl, $apiBase) {
                $fontUrl = $matches[1];
                $fontUrl = ltrim($fontUrl, './');
                if (strpos($fontUrl, 'http') !== 0) {
                    $fontUrl = $baseUrl . '/' . $fontUrl;
                }
                return 'url("' . $apiBase . '?route=/proxy/font&url=' . urlencode($fontUrl) . '")';
            },
            $data
        );
    }

    file_put_contents($cacheFile, $data);

    header('Content-Type: ' . $contentType);
    header('Cache-Control: public, max-age=604800');
    echo $data;
    exit;
}

if (isset($_GET['route'])) {
    $route = '/' . ltrim($_GET['route'], '/');
    
    if ($route === '/proxy/icons') {
        $file = $_GET['file'] ?? 'bold';
        $allowedFiles = [
            'regular' => 'https://cdn.jsdelivr.net/npm/@phosphor-icons/web@2.1.2/src/regular/style.css',
            'thin' => 'https://cdn.jsdelivr.net/npm/@phosphor-icons/web@2.1.2/src/thin/style.css',
            'light' => 'https://cdn.jsdelivr.net/npm/@phosphor-icons/web@2.1.2/src/light/style.css',
            'bold' => 'https://cdn.jsdelivr.net/npm/@phosphor-icons/web@2.1.2/src/bold/style.css',
            'fill' => 'https://cdn.jsdelivr.net/npm/@phosphor-icons/web@2.1.2/src/fill/style.css',
            'duotone' => 'https://cdn.jsdelivr.net/npm/@phosphor-icons/web@2.1.2/src/duotone/style.css',
        ];
        if (isset($allowedFiles[$file])) {
            proxyRequest($allowedFiles[$file], 'text/css', true);
        }
    } elseif ($route === '/proxy/font') {
        $url = $_GET['url'] ?? '';
        if (!empty($url)) {
            $url = urldecode($url);
            if (filter_var($url, FILTER_VALIDATE_URL)) {
                $ext = pathinfo(parse_url($url, PHP_URL_PATH), PATHINFO_EXTENSION);
                $contentTypes = [
                    'woff2' => 'font/woff2',
                    'woff' => 'font/woff',
                    'ttf' => 'font/ttf',
                    'otf' => 'font/otf',
                    'eot' => 'application/vnd.ms-fontobject',
                    'svg' => 'image/svg+xml',
                ];
                $contentType = $contentTypes[$ext] ?? 'application/octet-stream';
                proxyRequest($url, $contentType);
            }
        }
    } elseif ($route === '/proxy/sticker') {
        $name = $_GET['name'] ?? '';
        if (!empty($name)) {
            $name = preg_replace('/[^a-zA-Z0-9\x{4e00}-\x{9fa5}_-]/u', '', $name);
            if (!empty($name)) {
                $url = "https://cdn.jsdelivr.net/npm/sticker-heo@2022.7.5/Sticker-100/{$name}.png";
                proxyRequest($url, 'image/png');
            }
        }
    } elseif ($route === '/proxy/stickers') {
        proxyRequest('https://cdn.jsdelivr.net/npm/sticker-heo@2022.7.5/twikoo.json', 'application/json');
    }
}

header('Content-Type: application/json; charset=utf-8');

class CommentAPI
{
    private PDO $db;
    private array $config;
    private ?array $currentUser = null;

    public function __construct()
    {
        $this->config = require __DIR__ . '/config.php';
        $this->connectDB();
        $this->authenticate();
    }

    private function connectDB(): void
    {
        $d = $this->config['database'];
        $dsn = "mysql:host={$d['host']};port={$d['port']};dbname={$d['name']};charset={$d['charset']}";
        try {
            $this->db = new PDO($dsn, $d['user'], $d['pass'], [
                PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                PDO::ATTR_EMULATE_PREPARES => false,
            ]);
        } catch (PDOException $e) {
            $this->error('数据库连接失败', 500);
        }
    }

    private function authenticate(): void
    {
        $sessionKey = $_COOKIE['admin_session'] ?? $_SERVER['HTTP_AUTHORIZATION'] ?? null;
        if ($sessionKey) {
            $sessionKey = str_replace('Bearer ', '', $sessionKey);
            $stmt = $this->db->prepare('SELECT * FROM admin_sessions WHERE session_key = ? AND expires_at > NOW()');
            $stmt->execute([$sessionKey]);
            $session = $stmt->fetch();
            if ($session) {
                $this->currentUser = [
                    'id' => $session['admin_id'],
                    'role' => 'admin',
                ];
            }
        }
    }

    private function json(array $data, int $code = 200): void
    {
        http_response_code($code);
        echo json_encode($data, JSON_UNESCAPED_UNICODE);
        exit;
    }

    private function error(string $message, int $code = 400): void
    {
        $this->json(['error' => $message], $code);
    }

    private function success($data = null, string $message = '操作成功'): void
    {
        $this->json(['success' => true, 'message' => $message, 'data' => $data]);
    }

    private function requireAuth(): void
    {
        if (!$this->currentUser) {
            $this->error('请先登录', 401);
        }
    }

    private function getInput(): array
    {
        $input = file_get_contents('php://input');
        return json_decode($input, true) ?? $_POST;
    }

    private function getClientIP(): string
    {
        $headers = ['HTTP_CF_CONNECTING_IP', 'HTTP_X_FORWARDED_FOR', 'HTTP_X_REAL_IP', 'REMOTE_ADDR'];
        foreach ($headers as $header) {
            if (!empty($_SERVER[$header])) {
                $ip = $_SERVER[$header];
                if (strpos($ip, ',') !== false) {
                    $ip = trim(explode(',', $ip)[0]);
                }
                return filter_var($ip, FILTER_VALIDATE_IP) ?: $_SERVER['REMOTE_ADDR'];
            }
        }
        return $_SERVER['REMOTE_ADDR'] ?? '0.0.0.0';
    }

    private function generateToken(int $length = 32): string
    {
        return bin2hex(random_bytes($length));
    }

    private function sendMail(string $to, string $subject, string $body): bool
    {
        if (!$this->config['smtp']['enabled']) {
            return false;
        }
        $smtp = $this->config['smtp'];
        $headers = [
            'From: ' . mb_encode_mimeheader($smtp['from_name']) . ' <' . $smtp['from'] . '>',
            'Reply-To: ' . $smtp['from'],
            'MIME-Version: 1.0',
            'Content-Type: text/html; charset=UTF-8',
        ];
        return mail($to, mb_encode_mimeheader($subject), $body, implode("\r\n", $headers));
    }

    public function run(): void
    {
        $method = $_SERVER['REQUEST_METHOD'];
        
        if (isset($_GET['route'])) {
            $path = '/' . ltrim($_GET['route'], '/');
        } else {
            $path = parse_url($_SERVER['REQUEST_URI'], PHP_URL_PATH);
            $path = str_replace('/api.php', '', $path) ?: '/';
        }

        $routes = [
            'GET /comments' => 'listComments',
            'POST /comments' => 'createComment',
            'GET /comments/{id}' => 'getComment',
            'PUT /comments/{id}' => 'updateComment',
            'DELETE /comments/{id}' => 'deleteComment',
            'POST /comments/{id}/like' => 'likeComment',
            'POST /login' => 'login',
            'POST /logout' => 'logout',
            'GET /admin/comments' => 'adminListComments',
            'PUT /admin/comments/{id}/status' => 'updateCommentStatus',
            'DELETE /admin/comments/{id}' => 'adminDeleteComment',
            'GET /admin/stats' => 'getStats',
            'GET /admin/settings' => 'getSettings',
            'PUT /admin/settings' => 'updateSettings',
            'POST /admin/smtp-test' => 'testSmtp',
        ];

        foreach ($routes as $route => $handler) {
            $pattern = preg_replace('/\{[^}]+\}/', '([^/]+)', $route);
            $pattern = '#^' . $pattern . '$#';
            $routeMethod = strtok($route, ' ');
            $routePath = substr($route, strlen($routeMethod) + 1);

            if ($method === $routeMethod && preg_match($pattern, $path, $matches)) {
                array_shift($matches);
                call_user_func_array([$this, $handler], $matches);
                return;
            }
        }

        $this->error('接口不存在', 404);
    }

    public function listComments(): void
    {
        $pageId = $_GET['page_id'] ?? '';
        $page = max(1, (int)($_GET['page'] ?? 1));
        $limit = min(100, max(1, (int)($_GET['limit'] ?? $this->config['comment']['per_page'])));
        $offset = ($page - 1) * $limit;

        if (empty($pageId)) {
            $this->error('缺少页面标识');
        }

        $stmt = $this->db->prepare('
            SELECT c.*, 
                   (SELECT COUNT(*) FROM comment_likes WHERE comment_id = c.id) as like_count,
                   (SELECT COUNT(*) FROM comments WHERE parent_id = c.id AND status = "approved") as reply_count
            FROM comments c
            WHERE c.page_id = ? AND c.status = "approved" AND c.parent_id IS NULL
            ORDER BY c.created_at DESC
            LIMIT ? OFFSET ?
        ');
        $stmt->execute([$pageId, $limit, $offset]);
        $comments = $stmt->fetchAll();

        foreach ($comments as &$comment) {
            $stmt = $this->db->prepare('
                SELECT c.*,
                       (SELECT COUNT(*) FROM comment_likes WHERE comment_id = c.id) as like_count
                FROM comments c
                WHERE c.page_id = ? AND c.status = "approved" AND c.parent_id = ?
                ORDER BY c.created_at ASC
            ');
            $stmt->execute([$pageId, $comment['id']]);
            $comment['replies'] = $stmt->fetchAll();
        }

        $stmt = $this->db->prepare('SELECT COUNT(*) FROM comments WHERE page_id = ? AND status = "approved"');
        $stmt->execute([$pageId]);
        $total = (int)$stmt->fetchColumn();

        $this->success([
            'comments' => $comments,
            'pagination' => [
                'page' => $page,
                'limit' => $limit,
                'total' => $total,
                'pages' => ceil($total / $limit),
            ],
        ]);
    }

    public function createComment(): void
    {
        $input = $this->getInput();
        $pageId = trim($input['page_id'] ?? '');
        $parentId = $input['parent_id'] ?? null;
        $author = trim($input['author'] ?? '');
        $email = trim($input['email'] ?? '');
        $website = trim($input['website'] ?? '');
        $content = trim($input['content'] ?? '');

        if (empty($pageId) || empty($content)) {
            $this->error('页面标识和评论内容不能为空');
        }

        if (empty($email)) {
            $this->error('请输入邮箱');
        }

        if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
            $this->error('邮箱格式不正确');
        }

        if (mb_strlen($content) > $this->config['comment']['max_length']) {
            $this->error('评论内容过长');
        }

        if (!$this->config['comment']['allow_guest'] && empty($author)) {
            $this->error('请输入昵称');
        }

        if ($parentId) {
            $stmt = $this->db->prepare('SELECT id, author FROM comments WHERE id = ? AND page_id = ? AND status = "approved"');
            $stmt->execute([$parentId, $pageId]);
            if (!$stmt->fetch()) {
                $this->error('回复的评论不存在');
            }
        }

        $status = $this->config['comment']['require_approval'] ? 'pending' : 'approved';

        $stmt = $this->db->prepare('
            INSERT INTO comments (page_id, parent_id, author, email, website, content, ip, user_agent, status)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)
        ');
        $stmt->execute([
            $pageId,
            $parentId ?: null,
            $author ?: '匿名用户',
            $email ?: null,
            $website ?: null,
            $content,
            $this->getClientIP(),
            $_SERVER['HTTP_USER_AGENT'] ?? '',
            $status,
        ]);

        $commentId = (int)$this->db->lastInsertId();

        if ($this->config['comment']['notify_admin'] && $status === 'pending') {
            $subject = '[%s] 有新评论待审核';
            $body = '<p>页面 %s 收到新评论：</p><p>%s</p><p>请登录后台审核。</p>';
            $this->sendMail(
                $this->config['site']['admin_email'],
                sprintf($subject, $this->config['site']['name']),
                sprintf($body, $pageId, htmlspecialchars($content))
            );
        }

        $this->success([
            'id' => $commentId,
            'status' => $status,
        ], $status === 'approved' ? '评论成功' : '评论已提交，等待审核');
    }

    public function getComment(string $id): void
    {
        $stmt = $this->db->prepare('
            SELECT c.*, (SELECT COUNT(*) FROM comment_likes WHERE comment_id = c.id) as like_count
            FROM comments c
            WHERE c.id = ?
        ');
        $stmt->execute([(int)$id]);
        $comment = $stmt->fetch();

        if (!$comment) {
            $this->error('评论不存在', 404);
        }

        $this->success($comment);
    }

    public function updateComment(string $id): void
    {
        $input = $this->getInput();
        $content = trim($input['content'] ?? '');

        if (empty($content)) {
            $this->error('评论内容不能为空');
        }

        $stmt = $this->db->prepare('SELECT id, ip FROM comments WHERE id = ?');
        $stmt->execute([(int)$id]);
        $comment = $stmt->fetch();

        if (!$comment) {
            $this->error('评论不存在', 404);
        }

        if ($comment['ip'] !== $this->getClientIP() && !$this->currentUser) {
            $this->error('无权修改此评论', 403);
        }

        $stmt = $this->db->prepare('UPDATE comments SET content = ?, updated_at = NOW() WHERE id = ?');
        $stmt->execute([$content, (int)$id]);

        $this->success(null, '修改成功');
    }

    public function deleteComment(string $id): void
    {
        $stmt = $this->db->prepare('SELECT id, ip FROM comments WHERE id = ?');
        $stmt->execute([(int)$id]);
        $comment = $stmt->fetch();

        if (!$comment) {
            $this->error('评论不存在', 404);
        }

        if ($comment['ip'] !== $this->getClientIP() && !$this->currentUser) {
            $this->error('无权删除此评论', 403);
        }

        $stmt = $this->db->prepare('DELETE FROM comments WHERE id = ?');
        $stmt->execute([(int)$id]);

        $this->success(null, '删除成功');
    }

    public function likeComment(string $id): void
    {
        $ip = $this->getClientIP();
        $stmt = $this->db->prepare('SELECT id FROM comment_likes WHERE comment_id = ? AND ip = ?');
        $stmt->execute([(int)$id, $ip]);

        if ($stmt->fetch()) {
            $stmt = $this->db->prepare('DELETE FROM comment_likes WHERE comment_id = ? AND ip = ?');
            $stmt->execute([(int)$id, $ip]);
            $liked = false;
        } else {
            $stmt = $this->db->prepare('INSERT INTO comment_likes (comment_id, ip) VALUES (?, ?)');
            $stmt->execute([(int)$id, $ip]);
            $liked = true;
        }

        $stmt = $this->db->prepare('SELECT COUNT(*) FROM comment_likes WHERE comment_id = ?');
        $stmt->execute([(int)$id]);
        $count = (int)$stmt->fetchColumn();

        $this->success(['liked' => $liked, 'count' => $count]);
    }

    public function login(): void
    {
        $input = $this->getInput();
        $username = trim($input['username'] ?? '');
        $password = $input['password'] ?? '';

        if (empty($username) || empty($password)) {
            $this->error('请输入用户名和密码');
        }

        $stmt = $this->db->prepare('SELECT * FROM admins WHERE username = ?');
        $stmt->execute([$username]);
        $admin = $stmt->fetch();

        if (!$admin || !password_verify($password, $admin['password'])) {
            $this->error('用户名或密码错误');
        }

        $sessionKey = $this->generateToken();
        $expiresAt = date('Y-m-d H:i:s', strtotime('+7 days'));

        $stmt = $this->db->prepare('INSERT INTO admin_sessions (admin_id, session_key, expires_at) VALUES (?, ?, ?)');
        $stmt->execute([$admin['id'], $sessionKey, $expiresAt]);

        setcookie('admin_session', $sessionKey, [
            'expires' => strtotime('+7 days'),
            'path' => '/',
            'httponly' => true,
            'samesite' => 'Lax',
        ]);

        $this->success([
            'token' => $sessionKey,
            'admin' => [
                'id' => $admin['id'],
                'username' => $admin['username'],
            ],
        ], '登录成功');
    }

    public function logout(): void
    {
        $sessionKey = $_COOKIE['admin_session'] ?? $_SERVER['HTTP_AUTHORIZATION'] ?? null;
        if ($sessionKey) {
            $sessionKey = str_replace('Bearer ', '', $sessionKey);
            $stmt = $this->db->prepare('DELETE FROM admin_sessions WHERE session_key = ?');
            $stmt->execute([$sessionKey]);
        }

        setcookie('admin_session', '', [
            'expires' => time() - 3600,
            'path' => '/',
        ]);

        $this->success(null, '已退出登录');
    }

    public function adminListComments(): void
    {
        $this->requireAuth();

        $status = $_GET['status'] ?? '';
        $page = max(1, (int)($_GET['page'] ?? 1));
        $limit = min(100, max(1, (int)($_GET['limit'] ?? 20)));
        $offset = ($page - 1) * $limit;

        $where = '1=1';
        $params = [];

        if ($status && in_array($status, ['pending', 'approved', 'spam'])) {
            $where .= ' AND status = ?';
            $params[] = $status;
        }

        $stmt = $this->db->prepare("
            SELECT c.*,
                   (SELECT COUNT(*) FROM comment_likes WHERE comment_id = c.id) as like_count
            FROM comments c
            WHERE {$where}
            ORDER BY c.created_at DESC
            LIMIT ? OFFSET ?
        ");
        foreach ($params as $i => $param) {
            $stmt->bindValue($i + 1, $param);
        }
        $stmt->bindValue(count($params) + 1, $limit, PDO::PARAM_INT);
        $stmt->bindValue(count($params) + 2, $offset, PDO::PARAM_INT);
        $stmt->execute();
        $comments = $stmt->fetchAll();

        $stmt = $this->db->prepare("SELECT COUNT(*) FROM comments WHERE {$where}");
        foreach ($params as $i => $param) {
            $stmt->bindValue($i + 1, $param);
        }
        $stmt->execute();
        $total = (int)$stmt->fetchColumn();

        $this->success([
            'comments' => $comments,
            'pagination' => [
                'page' => $page,
                'limit' => $limit,
                'total' => $total,
                'pages' => ceil($total / $limit),
            ],
        ]);
    }

    public function updateCommentStatus(string $id): void
    {
        $this->requireAuth();

        $input = $this->getInput();
        $status = $input['status'] ?? '';

        if (!in_array($status, ['pending', 'approved', 'spam'])) {
            $this->error('无效的状态');
        }

        $stmt = $this->db->prepare('UPDATE comments SET status = ? WHERE id = ?');
        $stmt->execute([$status, (int)$id]);

        $this->success(null, '状态已更新');
    }

    public function adminDeleteComment(string $id): void
    {
        $this->requireAuth();

        $stmt = $this->db->prepare('DELETE FROM comments WHERE id = ?');
        $stmt->execute([(int)$id]);

        $this->success(null, '删除成功');
    }

    public function getStats(): void
    {
        $this->requireAuth();

        $stats = [];

        $stmt = $this->db->query('SELECT COUNT(*) FROM comments');
        $stats['total'] = (int)$stmt->fetchColumn();

        $stmt = $this->db->query('SELECT COUNT(*) FROM comments WHERE status = "pending"');
        $stats['pending'] = (int)$stmt->fetchColumn();

        $stmt = $this->db->query('SELECT COUNT(*) FROM comments WHERE status = "approved"');
        $stats['approved'] = (int)$stmt->fetchColumn();

        $stmt = $this->db->query('SELECT COUNT(*) FROM comments WHERE status = "spam"');
        $stats['spam'] = (int)$stmt->fetchColumn();

        $stmt = $this->db->query('SELECT COUNT(*) FROM comment_likes');
        $stats['likes'] = (int)$stmt->fetchColumn();

        $stmt = $this->db->query('SELECT COUNT(DISTINCT page_id) FROM comments');
        $stats['pages'] = (int)$stmt->fetchColumn();

        $stmt = $this->db->query('SELECT DATE(created_at) as date, COUNT(*) as count FROM comments WHERE created_at > DATE_SUB(NOW(), INTERVAL 7 DAY) GROUP BY DATE(created_at) ORDER BY date');
        $stats['recent'] = $stmt->fetchAll();

        $this->success($stats);
    }

    public function getSettings(): void
    {
        $this->requireAuth();
        $this->success($this->config);
    }

    public function updateSettings(): void
    {
        $this->requireAuth();
        $input = $this->getInput();
        
        $configFile = __DIR__ . '/config.php';
        $newConfig = $this->config;
        
        if (isset($input['smtp']) && is_array($input['smtp'])) {
            $smtp = $input['smtp'];
            $newConfig['smtp'] = [
                'enabled' => !empty($smtp['enabled']),
                'host' => trim($smtp['host'] ?? ''),
                'port' => (int)($smtp['port'] ?? 587),
                'user' => trim($smtp['user'] ?? ''),
                'pass' => $smtp['pass'] ?? '',
                'from' => trim($smtp['from'] ?? ''),
                'from_name' => trim($smtp['from_name'] ?? '冰裂纹 Crackle'),
            ];
        }
        
        if (isset($input['site']) && is_array($input['site'])) {
            $site = $input['site'];
            $newConfig['site']['name'] = trim($site['name'] ?? $newConfig['site']['name']);
            $newConfig['site']['url'] = trim($site['url'] ?? $newConfig['site']['url']);
            $newConfig['site']['admin_email'] = trim($site['admin_email'] ?? $newConfig['site']['admin_email']);
        }
        
        if (isset($input['comment']) && is_array($input['comment'])) {
            $comment = $input['comment'];
            $newConfig['comment']['require_approval'] = !empty($comment['require_approval']);
            $newConfig['comment']['max_length'] = max(100, min(10000, (int)($comment['max_length'] ?? 2000)));
            $newConfig['comment']['per_page'] = max(5, min(100, (int)($comment['per_page'] ?? 20)));
            $newConfig['comment']['allow_guest'] = !empty($comment['allow_guest']);
            $newConfig['comment']['notify_admin'] = !empty($comment['notify_admin']);
            $newConfig['comment']['notify_reply'] = !empty($comment['notify_reply']);
        }
        
        $configContent = "<?php\ndeclare(strict_types=1);\n\nreturn " . var_export($newConfig, true) . ";\n";
        
        if (file_put_contents($configFile, $configContent) === false) {
            $this->error('配置保存失败，请检查文件权限');
        }
        
        $this->success(['message' => '配置已保存']);
    }

    public function testSmtp(): void
    {
        $this->requireAuth();
        $input = $this->getInput();
        
        $host = trim($input['host'] ?? '');
        $port = (int)($input['port'] ?? 587);
        $user = trim($input['user'] ?? '');
        $pass = $input['pass'] ?? '';
        $from = trim($input['from'] ?? '');
        $to = trim($input['to'] ?? $from);
        
        if (empty($host) || empty($port)) {
            $this->error('请填写SMTP服务器和端口');
        }
        
        if (empty($to)) {
            $this->error('请填写测试收件邮箱');
        }
        
        $errno = 0;
        $errstr = '';
        $socket = @fsockopen($host, $port, $errno, $errstr, 10);
        
        if (!$socket) {
            $this->error("无法连接到 {$host}:{$port} - {$errstr}");
        }
        
        $read = function($socket) {
            $data = '';
            while ($line = fgets($socket, 515)) {
                $data .= $line;
                if (substr($line, 3, 1) === ' ') {
                    break;
                }
            }
            return $data;
        };
        
        $response = $read($socket);
        if (substr($response, 0, 3) !== '220') {
            fclose($socket);
            $this->error('服务器响应异常: ' . trim($response));
        }
        
        $localhost = $_SERVER['SERVER_NAME'] ?? 'localhost';
        fwrite($socket, "EHLO {$localhost}\r\n");
        $response = $read($socket);
        if (substr($response, 0, 3) !== '250') {
            fclose($socket);
            $this->error('EHLO 失败: ' . trim($response));
        }
        
        fwrite($socket, "STARTTLS\r\n");
        $response = $read($socket);
        if (substr($response, 0, 3) === '220') {
            stream_socket_enable_crypto($socket, true, STREAM_CRYPTO_METHOD_TLS_CLIENT);
            fwrite($socket, "EHLO {$localhost}\r\n");
            $response = $read($socket);
        }
        
        if (!empty($user) && !empty($pass)) {
            fwrite($socket, "AUTH LOGIN\r\n");
            $response = $read($socket);
            if (substr($response, 0, 3) !== '334') {
                fclose($socket);
                $this->error('AUTH LOGIN 失败: ' . trim($response));
            }
            
            fwrite($socket, base64_encode($user) . "\r\n");
            $response = $read($socket);
            if (substr($response, 0, 3) !== '334') {
                fclose($socket);
                $this->error('用户名验证失败: ' . trim($response));
            }
            
            fwrite($socket, base64_encode($pass) . "\r\n");
            $response = $read($socket);
            if (substr($response, 0, 3) !== '235') {
                fclose($socket);
                $this->error('密码验证失败: ' . trim($response));
            }
        }
        
        fwrite($socket, "MAIL FROM:<{$from}>\r\n");
        $response = $read($socket);
        if (substr($response, 0, 3) !== '250') {
            fclose($socket);
            $this->error('MAIL FROM 失败: ' . trim($response));
        }
        
        fwrite($socket, "RCPT TO:<{$to}>\r\n");
        $response = $read($socket);
        if (substr($response, 0, 3) !== '250') {
            fclose($socket);
            $this->error('RCPT TO 失败: ' . trim($response));
        }
        
        fwrite($socket, "DATA\r\n");
        $response = $read($socket);
        if (substr($response, 0, 3) !== '354') {
            fclose($socket);
            $this->error('DATA 失败: ' . trim($response));
        }
        
        $message = "From: {$from}\r\n";
        $message .= "To: {$to}\r\n";
        $message .= "Subject: =?UTF-8?B?" . base64_encode('[冰裂纹] SMTP测试邮件') . "?=\r\n";
        $message .= "MIME-Version: 1.0\r\n";
        $message .= "Content-Type: text/html; charset=UTF-8\r\n";
        $message .= "\r\n";
        $message .= '<p>这是一封SMTP测试邮件，如果您收到此邮件，说明SMTP配置正确。</p>';
        $message .= '<p>发送时间: ' . date('Y-m-d H:i:s') . '</p>';
        $message .= "\r\n.\r\n";
        
        fwrite($socket, $message);
        $response = $read($socket);
        if (substr($response, 0, 3) !== '250') {
            fclose($socket);
            $this->error('邮件发送失败: ' . trim($response));
        }
        
        fwrite($socket, "QUIT\r\n");
        fclose($socket);
        
        $this->success(['message' => "测试邮件已发送至 {$to}，请检查收件箱"]);
    }
}

$api = new CommentAPI();
$api->run();
