<?php
declare(strict_types=1);

$installedFile = __DIR__ . '/.installed';
if (!file_exists($installedFile)) {
    header('Location: install.php');
    exit;
}

$config = require __DIR__ . '/config.php';
$d = $config['database'];
$dsn = "mysql:host={$d['host']};port={$d['port']};dbname={$d['name']};charset={$d['charset']}";

try {
    $db = new PDO($dsn, $d['user'], $d['pass'], [
        PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
        PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
    ]);
} catch (PDOException $e) {
    die('数据库连接失败');
}

function isLoggedIn(PDO $db): bool
{
    $sessionKey = $_COOKIE['admin_session'] ?? null;
    if (!$sessionKey) return false;

    $stmt = $db->prepare('SELECT id FROM admin_sessions WHERE session_key = ? AND expires_at > NOW()');
    $stmt->execute([$sessionKey]);
    return (bool)$stmt->fetch();
}

function checkAuth(PDO $db): void
{
    if (!isLoggedIn($db)) {
        header('Location: admin.php?action=login');
        exit;
    }
}

$action = $_GET['action'] ?? 'dashboard';

if ($action === 'login') {
    $error = '';
    if ($_SERVER['REQUEST_METHOD'] === 'POST') {
        $username = trim($_POST['username'] ?? '');
        $password = $_POST['password'] ?? '';

        $stmt = $db->prepare('SELECT id, password FROM admins WHERE username = ?');
        $stmt->execute([$username]);
        $admin = $stmt->fetch();

        if ($admin && password_verify($password, $admin['password'])) {
            $sessionKey = bin2hex(random_bytes(32));
            $expiresAt = date('Y-m-d H:i:s', strtotime('+7 days'));

            $stmt = $db->prepare('INSERT INTO admin_sessions (admin_id, session_key, expires_at) VALUES (?, ?, ?)');
            $stmt->execute([$admin['id'], $sessionKey, $expiresAt]);

            setcookie('admin_session', $sessionKey, [
                'expires' => strtotime('+7 days'),
                'path' => '/',
                'httponly' => true,
                'samesite' => 'Lax',
            ]);

            header('Location: admin.php');
            exit;
        }
        $error = '用户名或密码错误';
    }
} elseif ($action === 'logout') {
    $sessionKey = $_COOKIE['admin_session'] ?? null;
    if ($sessionKey) {
        $stmt = $db->prepare('DELETE FROM admin_sessions WHERE session_key = ?');
        $stmt->execute([$sessionKey]);
    }
    setcookie('admin_session', '', ['expires' => time() - 3600, 'path' => '/']);
    header('Location: admin.php?action=login');
    exit;
} else {
    checkAuth($db);
}

$stats = [];
if ($action === 'dashboard' && isLoggedIn($db)) {
    $stmt = $db->query('SELECT COUNT(*) FROM comments');
    $stats['total'] = (int)$stmt->fetchColumn();

    $stmt = $db->query('SELECT COUNT(*) FROM comments WHERE status = "pending"');
    $stats['pending'] = (int)$stmt->fetchColumn();

    $stmt = $db->query('SELECT COUNT(*) FROM comments WHERE status = "approved"');
    $stats['approved'] = (int)$stmt->fetchColumn();

    $stmt = $db->query('SELECT COUNT(*) FROM comments WHERE status = "spam"');
    $stats['spam'] = (int)$stmt->fetchColumn();

    $stmt = $db->query('SELECT COUNT(*) FROM comment_likes');
    $stats['likes'] = (int)$stmt->fetchColumn();

    $stmt = $db->query('SELECT COUNT(DISTINCT page_id) FROM comments');
    $stats['pages'] = (int)$stmt->fetchColumn();
}

$comments = [];
if ($action === 'comments' && isLoggedIn($db)) {
    $status = $_GET['status'] ?? '';
    $page = max(1, (int)($_GET['page'] ?? 1));
    $limit = 20;
    $offset = ($page - 1) * $limit;

    $where = '1=1';
    $params = [];

    if ($status && in_array($status, ['pending', 'approved', 'spam'])) {
        $where .= ' AND status = ?';
        $params[] = $status;
    }

    $stmt = $db->prepare("SELECT * FROM comments WHERE {$where} ORDER BY created_at DESC LIMIT {$limit} OFFSET {$offset}");
    $stmt->execute($params);
    $comments = $stmt->fetchAll();

    $stmt = $db->prepare("SELECT COUNT(*) FROM comments WHERE {$where}");
    $stmt->execute($params);
    $total = (int)$stmt->fetchColumn();
    $totalPages = ceil($total / $limit);
}
?>
<!DOCTYPE html>
<html lang="zh-CN">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= htmlspecialchars($config['site']['name']) ?> - 管理后台</title>
    <link rel="stylesheet" href="https://img.penlife.cn/font/OPPO%20Sans%204/result.css">
    <script src="https://unpkg.com/@phosphor-icons/web"></script>
    <link rel="stylesheet" href="assets/css/main.css">
</head>
<body class="admin-page">
    <?php if ($action === 'login'): ?>
        <div class="login-container">
            <div class="login-card">
                <h1>登录</h1>
                <?php if ($error): ?>
                    <div class="alert alert-error"><?= htmlspecialchars($error) ?></div>
                <?php endif; ?>
                <form method="post">
                    <div class="form-group">
                        <label>用户名</label>
                        <input type="text" name="username" required autofocus>
                    </div>
                    <div class="form-group">
                        <label>密码</label>
                        <input type="password" name="password" required>
                    </div>
                    <button type="submit" class="btn btn-primary">登录</button>
                </form>
            </div>
        </div>
    <?php else: ?>
        <div class="admin-layout">
            <aside class="sidebar">
                <div class="sidebar-header">
                    <h1><?= htmlspecialchars($config['site']['name']) ?></h1>
                </div>
                <nav class="sidebar-nav">
                    <a href="admin.php" class="nav-item <?= $action === 'dashboard' ? 'active' : '' ?>">
                        <i class="ph-bold ph-house"></i>
                        <span>概览</span>
                    </a>
                    <a href="admin.php?action=comments" class="nav-item <?= $action === 'comments' ? 'active' : '' ?>">
                        <i class="ph-bold ph-chat-circle-text"></i>
                        <span>评论管理</span>
                    </a>
                    <a href="admin.php?action=comments&status=pending" class="nav-item">
                        <i class="ph-bold ph-clock-counter-clockwise"></i>
                        <span>待审核</span>
                        <?php if ($stats['pending'] ?? 0): ?>
                            <span class="badge"><?= $stats['pending'] ?></span>
                        <?php endif; ?>
                    </a>
                    <a href="admin.php?action=settings" class="nav-item <?= $action === 'settings' ? 'active' : '' ?>">
                        <i class="ph-bold ph-gear"></i>
                        <span>设置</span>
                    </a>
                </nav>
                <div class="sidebar-footer">
                    <a href="admin.php?action=logout" class="nav-item">
                        <i class="ph-bold ph-sign-out"></i>
                        <span>退出登录</span>
                    </a>
                </div>
            </aside>

            <main class="main-content">
                <?php if ($action === 'dashboard'): ?>
                    <div class="page-header">
                        <h2>概览</h2>
                    </div>

                    <div class="stats-grid">
                        <div class="stat-card">
                            <div class="stat-icon">
                                <i class="ph-bold ph-chat-circle-text"></i>
                            </div>
                            <div class="stat-info">
                                <span class="stat-value"><?= $stats['total'] ?></span>
                                <span class="stat-label">总评论</span>
                            </div>
                        </div>
                        <div class="stat-card">
                            <div class="stat-icon approved">
                                <i class="ph-bold ph-check-circle"></i>
                            </div>
                            <div class="stat-info">
                                <span class="stat-value"><?= $stats['approved'] ?></span>
                                <span class="stat-label">已通过</span>
                            </div>
                        </div>
                        <div class="stat-card">
                            <div class="stat-icon pending">
                                <i class="ph-bold ph-clock-counter-clockwise"></i>
                            </div>
                            <div class="stat-info">
                                <span class="stat-value"><?= $stats['pending'] ?></span>
                                <span class="stat-label">待审核</span>
                            </div>
                        </div>
                        <div class="stat-card">
                            <div class="stat-icon spam">
                                <i class="ph-bold ph-prohibit"></i>
                            </div>
                            <div class="stat-info">
                                <span class="stat-value"><?= $stats['spam'] ?></span>
                                <span class="stat-label">垃圾评论</span>
                            </div>
                        </div>
                        <div class="stat-card">
                            <div class="stat-icon likes">
                                <i class="ph-bold ph-heart"></i>
                            </div>
                            <div class="stat-info">
                                <span class="stat-value"><?= $stats['likes'] ?></span>
                                <span class="stat-label">点赞数</span>
                            </div>
                        </div>
                        <div class="stat-card">
                            <div class="stat-icon pages">
                                <i class="ph-bold ph-file-text"></i>
                            </div>
                            <div class="stat-info">
                                <span class="stat-value"><?= $stats['pages'] ?></span>
                                <span class="stat-label">页面数</span>
                            </div>
                        </div>
                    </div>

                <?php elseif ($action === 'comments'): ?>
                    <div class="page-header">
                        <h2>评论管理</h2>
                        <div class="filter-tabs">
                            <a href="admin.php?action=comments" class="filter-tab <?= !$status ? 'active' : '' ?>">全部</a>
                            <a href="admin.php?action=comments&status=pending" class="filter-tab <?= $status === 'pending' ? 'active' : '' ?>">待审核</a>
                            <a href="admin.php?action=comments&status=approved" class="filter-tab <?= $status === 'approved' ? 'active' : '' ?>">已通过</a>
                            <a href="admin.php?action=comments&status=spam" class="filter-tab <?= $status === 'spam' ? 'active' : '' ?>">垃圾</a>
                        </div>
                    </div>

                    <div class="comments-list">
                        <?php if (empty($comments)): ?>
                            <div class="empty-state">
                                <i class="ph-bold ph-chat-circle-text"></i>
                                <p>暂无评论</p>
                            </div>
                        <?php else: ?>
                            <?php foreach ($comments as $comment): ?>
                                <div class="comment-item" data-id="<?= $comment['id'] ?>">
                                    <div class="comment-header">
                                        <span class="comment-author">
                                            <?= htmlspecialchars($comment['author']) ?>
                                            <?php if ($comment['email']): ?>
                                                <span class="comment-email"><?= htmlspecialchars($comment['email']) ?></span>
                                            <?php endif; ?>
                                        </span>
                                        <span class="comment-status status-<?= $comment['status'] ?>">
                                            <?= ['pending' => '待审核', 'approved' => '已通过', 'spam' => '垃圾'][$comment['status']] ?>
                                        </span>
                                    </div>
                                    <div class="comment-content">
                                        <?= nl2br(htmlspecialchars($comment['content'])) ?>
                                    </div>
                                    <div class="comment-meta">
                                        <span><i class="ph-bold ph-file-text"></i> <?= htmlspecialchars($comment['page_id']) ?></span>
                                        <span><i class="ph-bold ph-clock"></i> <?= $comment['created_at'] ?></span>
                                        <span><i class="ph-bold ph-ip-address"></i> <?= htmlspecialchars($comment['ip']) ?></span>
                                    </div>
                                    <div class="comment-actions">
                                        <?php if ($comment['status'] !== 'approved'): ?>
                                            <button class="btn btn-sm btn-success" onclick="updateStatus(<?= $comment['id'] ?>, 'approved')">
                                                <i class="ph-bold ph-check"></i> 通过
                                            </button>
                                        <?php endif; ?>
                                        <?php if ($comment['status'] !== 'spam'): ?>
                                            <button class="btn btn-sm btn-warning" onclick="updateStatus(<?= $comment['id'] ?>, 'spam')">
                                                <i class="ph-bold ph-prohibit"></i> 标记垃圾
                                            </button>
                                        <?php endif; ?>
                                        <button class="btn btn-sm btn-danger" onclick="deleteComment(<?= $comment['id'] ?>)">
                                            <i class="ph-bold ph-trash"></i> 删除
                                        </button>
                                    </div>
                                </div>
                            <?php endforeach; ?>

                            <?php if ($totalPages > 1): ?>
                                <div class="pagination">
                                    <?php if ($page > 1): ?>
                                        <a href="?action=comments&status=<?= $status ?>&page=<?= $page - 1 ?>" class="page-btn">上一页</a>
                                    <?php endif; ?>
                                    <span class="page-info">第 <?= $page ?> / <?= $totalPages ?> 页</span>
                                    <?php if ($page < $totalPages): ?>
                                        <a href="?action=comments&status=<?= $status ?>&page=<?= $page + 1 ?>" class="page-btn">下一页</a>
                                    <?php endif; ?>
                                </div>
                            <?php endif; ?>
                        <?php endif; ?>
                    </div>
                <?php elseif ($action === 'settings'): ?>
                    <div class="page-header">
                        <h2>系统设置</h2>
                    </div>

                    <div class="settings-container">
                        <div class="settings-section">
                            <h3><i class="ph-bold ph-envelope-simple"></i> SMTP 邮件配置</h3>
                            <form id="smtp-form" class="settings-form">
                                <div class="form-row">
                                    <label class="checkbox-label">
                                        <input type="checkbox" name="smtp_enabled" id="smtp_enabled">
                                        <span>启用 SMTP 邮件通知</span>
                                    </label>
                                </div>
                                <div class="form-row">
                                    <div class="form-group">
                                        <label>SMTP 服务器</label>
                                        <input type="text" name="smtp_host" id="smtp_host" placeholder="smtp.example.com">
                                    </div>
                                    <div class="form-group">
                                        <label>端口</label>
                                        <input type="number" name="smtp_port" id="smtp_port" placeholder="587">
                                    </div>
                                </div>
                                <div class="form-row">
                                    <div class="form-group">
                                        <label>用户名</label>
                                        <input type="text" name="smtp_user" id="smtp_user" placeholder="your@email.com">
                                    </div>
                                    <div class="form-group">
                                        <label>密码</label>
                                        <input type="password" name="smtp_pass" id="smtp_pass" placeholder="••••••••">
                                    </div>
                                </div>
                                <div class="form-row">
                                    <div class="form-group">
                                        <label>发件人邮箱</label>
                                        <input type="email" name="smtp_from" id="smtp_from" placeholder="noreply@example.com">
                                    </div>
                                    <div class="form-group">
                                        <label>发件人名称</label>
                                        <input type="text" name="smtp_from_name" id="smtp_from_name" placeholder="冰裂纹 Crackle">
                                    </div>
                                </div>
                                <div class="form-row">
                                    <div class="form-group">
                                        <label>测试收件邮箱</label>
                                        <input type="email" name="smtp_test_to" id="smtp_test_to" placeholder="测试邮件接收地址">
                                    </div>
                                    <div class="form-group">
                                        <label>&nbsp;</label>
                                        <button type="button" class="btn btn-outline" onclick="testSmtp()">
                                            <i class="ph-bold ph-paper-plane-tilt"></i> 发送测试邮件
                                        </button>
                                    </div>
                                </div>
                                <div id="smtp-test-result" class="smtp-test-result"></div>
                            </form>
                        </div>

                        <div class="settings-section">
                            <h3><i class="ph-bold ph-globe"></i> 站点设置</h3>
                            <form id="site-form" class="settings-form">
                                <div class="form-row">
                                    <div class="form-group">
                                        <label>站点名称</label>
                                        <input type="text" name="site_name" id="site_name">
                                    </div>
                                    <div class="form-group">
                                        <label>站点 URL</label>
                                        <input type="url" name="site_url" id="site_url" placeholder="https://example.com">
                                    </div>
                                </div>
                                <div class="form-row">
                                    <div class="form-group">
                                        <label>管理员邮箱</label>
                                        <input type="email" name="site_admin_email" id="site_admin_email" placeholder="admin@example.com">
                                    </div>
                                </div>
                            </form>
                        </div>

                        <div class="settings-section">
                            <h3><i class="ph-bold ph-chat-circle-text"></i> 评论设置</h3>
                            <form id="comment-form" class="settings-form">
                                <div class="form-row checkbox-row">
                                    <label class="checkbox-label">
                                        <input type="checkbox" name="comment_require_approval" id="comment_require_approval">
                                        <span>评论需要审核</span>
                                    </label>
                                    <label class="checkbox-label">
                                        <input type="checkbox" name="comment_allow_guest" id="comment_allow_guest">
                                        <span>允许游客评论</span>
                                    </label>
                                </div>
                                <div class="form-row checkbox-row">
                                    <label class="checkbox-label">
                                        <input type="checkbox" name="comment_notify_admin" id="comment_notify_admin">
                                        <span>新评论通知管理员</span>
                                    </label>
                                    <label class="checkbox-label">
                                        <input type="checkbox" name="comment_notify_reply" id="comment_notify_reply">
                                        <span>回复通知用户</span>
                                    </label>
                                </div>
                                <div class="form-row">
                                    <div class="form-group">
                                        <label>评论最大长度</label>
                                        <input type="number" name="comment_max_length" id="comment_max_length" min="100" max="10000">
                                    </div>
                                    <div class="form-group">
                                        <label>每页显示条数</label>
                                        <input type="number" name="comment_per_page" id="comment_per_page" min="5" max="100">
                                    </div>
                                </div>
                            </form>
                        </div>

                        <div class="settings-actions">
                            <button type="button" class="btn btn-primary" onclick="saveSettings()">
                                <i class="ph-bold ph-check"></i> 保存设置
                            </button>
                            <span id="save-status" class="save-status"></span>
                        </div>
                    </div>
                <?php endif; ?>
            </main>
        </div>
    <?php endif; ?>

    <script>
        const apiBase = 'api.php';

        async function api(endpoint, method = 'GET', data = null) {
            const options = {
                method,
                headers: { 'Content-Type': 'application/json' },
            };
            if (data) options.body = JSON.stringify(data);
            const res = await fetch(apiBase + endpoint, options);
            return res.json();
        }

        async function updateStatus(id, status) {
            if (!confirm('确定要更改此评论的状态吗？')) return;
            try {
                const result = await api(`/admin/comments/${id}/status`, 'PUT', { status });
                if (result.success) location.reload();
                else alert(result.error);
            } catch (e) {
                alert('操作失败');
            }
        }

        async function deleteComment(id) {
            if (!confirm('确定要删除此评论吗？')) return;
            try {
                const result = await api(`/admin/comments/${id}`, 'DELETE');
                if (result.success) location.reload();
                else alert(result.error);
            } catch (e) {
                alert('操作失败');
            }
        }

        let settings = null;

        async function loadSettings() {
            try {
                const result = await api('/admin/settings');
                if (result.success) {
                    settings = result.data;
                    populateSettings();
                }
            } catch (e) {
                console.error('加载设置失败', e);
            }
        }

        function populateSettings() {
            if (!settings) return;
            
            document.getElementById('smtp_enabled').checked = settings.smtp.enabled;
            document.getElementById('smtp_host').value = settings.smtp.host || '';
            document.getElementById('smtp_port').value = settings.smtp.port || 587;
            document.getElementById('smtp_user').value = settings.smtp.user || '';
            document.getElementById('smtp_pass').value = settings.smtp.pass || '';
            document.getElementById('smtp_from').value = settings.smtp.from || '';
            document.getElementById('smtp_from_name').value = settings.smtp.from_name || '';
            
            document.getElementById('site_name').value = settings.site.name || '';
            document.getElementById('site_url').value = settings.site.url || '';
            document.getElementById('site_admin_email').value = settings.site.admin_email || '';
            
            document.getElementById('comment_require_approval').checked = settings.comment.require_approval;
            document.getElementById('comment_allow_guest').checked = settings.comment.allow_guest;
            document.getElementById('comment_notify_admin').checked = settings.comment.notify_admin;
            document.getElementById('comment_notify_reply').checked = settings.comment.notify_reply;
            document.getElementById('comment_max_length').value = settings.comment.max_length || 2000;
            document.getElementById('comment_per_page').value = settings.comment.per_page || 20;
        }

        async function saveSettings() {
            const statusEl = document.getElementById('save-status');
            statusEl.textContent = '保存中...';
            statusEl.className = 'save-status';
            
            const data = {
                smtp: {
                    enabled: document.getElementById('smtp_enabled').checked,
                    host: document.getElementById('smtp_host').value,
                    port: parseInt(document.getElementById('smtp_port').value) || 587,
                    user: document.getElementById('smtp_user').value,
                    pass: document.getElementById('smtp_pass').value,
                    from: document.getElementById('smtp_from').value,
                    from_name: document.getElementById('smtp_from_name').value,
                },
                site: {
                    name: document.getElementById('site_name').value,
                    url: document.getElementById('site_url').value,
                    admin_email: document.getElementById('site_admin_email').value,
                },
                comment: {
                    require_approval: document.getElementById('comment_require_approval').checked,
                    allow_guest: document.getElementById('comment_allow_guest').checked,
                    notify_admin: document.getElementById('comment_notify_admin').checked,
                    notify_reply: document.getElementById('comment_notify_reply').checked,
                    max_length: parseInt(document.getElementById('comment_max_length').value) || 2000,
                    per_page: parseInt(document.getElementById('comment_per_page').value) || 20,
                },
            };
            
            try {
                const result = await api('/admin/settings', 'PUT', data);
                if (result.success) {
                    statusEl.textContent = '保存成功';
                    statusEl.className = 'save-status success';
                    setTimeout(() => { statusEl.textContent = ''; }, 3000);
                } else {
                    statusEl.textContent = result.error || '保存失败';
                    statusEl.className = 'save-status error';
                }
            } catch (e) {
                statusEl.textContent = '保存失败';
                statusEl.className = 'save-status error';
            }
        }

        async function testSmtp() {
            const resultEl = document.getElementById('smtp-test-result');
            resultEl.className = 'smtp-test-result';
            resultEl.textContent = '正在测试连接...';
            
            const data = {
                host: document.getElementById('smtp_host').value,
                port: parseInt(document.getElementById('smtp_port').value) || 587,
                user: document.getElementById('smtp_user').value,
                pass: document.getElementById('smtp_pass').value,
                from: document.getElementById('smtp_from').value,
                to: document.getElementById('smtp_test_to').value || document.getElementById('smtp_from').value,
            };
            
            if (!data.host) {
                resultEl.className = 'smtp-test-result error';
                resultEl.textContent = '请填写SMTP服务器';
                return;
            }
            
            try {
                const result = await api('/admin/smtp-test', 'POST', data);
                if (result.success) {
                    resultEl.className = 'smtp-test-result success';
                    resultEl.textContent = result.data.message;
                } else {
                    resultEl.className = 'smtp-test-result error';
                    resultEl.textContent = result.error || '测试失败';
                }
            } catch (e) {
                resultEl.className = 'smtp-test-result error';
                resultEl.textContent = '测试失败: ' + e.message;
            }
        }

        if (document.querySelector('.settings-container')) {
            loadSettings();
        }
    </script>
</body>
</html>
